/* (C) 2005 by Marcin Wiacek www.mwiacek.com */

#ifndef gsm_sms_h
#define gsm_sms_h

#ifdef WIN32
#  include <windows.h>
#endif

#include "../gsmmisc.h"
#include "../../misc/misc.h"

GSM_Error 	GSM_DecodeSMSNumber(wchar_t *Destination, const unsigned char *Source, int len, BOOLEAN semioctets);
GSM_Error 	GSM_EncodeSMSNumber(unsignedstring *Destination, wchar_t *Source, BOOLEAN semioctets);
GSM_Error 	GSM_DecodeSMSText(wchar_t *Destination, const unsigned char *Source, int len, int len3, int udhlen);
GSM_Error 	GSM_EncodeSMSText(unsigned char *Destination, const wchar_t *Source, int *GetFromSRC, int *SavedInSMS, int MaxInSMS, int *UsedInSMS, int udhlen);

typedef enum {
	SMS_UDH_None=1,
	SMS_UDH_Linked_Short, //8bit
	SMS_UDH_Linked_Long,    //16bit
	SMS_UDH_Fax,
	SMS_UDH_Voice,
	SMS_UDH_Email,
	SMS_UDH_Other,
	SMS_UDH_NokiaRingtone,
	SMS_UDH_NokiaOperatorLogo,
	SMS_UDH_NokiaCallerLogo,
	SMS_UDH_NokiaWAP
} SMS_UDH_Type;

typedef struct {
	SMS_UDH_Type 	Type;
	int		Length;
	char   		Text[10];
	int	     	ID8bit;
	long	     	ID16bit;
	int	     	PartNumber8bit;
	int	     	AllParts8bit;
} GSM_SMS_UDH_Types;

static GSM_SMS_UDH_Types UDHHeaders[] = {

/* See GSM 03.40 section 9.2.3.24.1
 * 1 byte 0x00 : IEI concatenated short messages, 8-bit reference number
 * 1 byte 0x03 : length of rest
 * 1 byte 0x01 : unique ID for message series
 * 1 byte 0x00 : how many SMS in sequence
 * 1 byte 0x00 : number of current SMS in sequence
 */
{SMS_UDH_Linked_Short,		0x05,"\x00\x03\x00\x00\x00",2,-1,4,3},

/* See GSM 03.40 section 9.2.3.24.2 (Special SMS Message Indication)
 * 1 byte 0x01 :
 * 1 byte 0x02 : length of rest
 * 1 byte      : type
 * 1 byte      : msg num
 */
{SMS_UDH_Fax,         		0x04,"\x01\x02\x01\x00",-1,-1,-1,-1},
{SMS_UDH_Voice,       		0x04,"\x01\x02\x00\x00",-1,-1,-1,-1},
{SMS_UDH_Email,       		0x04,"\x01\x02\x02\x00",-1,-1,-1,-1},
{SMS_UDH_Other,       		0x04,"\x01\x02\x03\x00",-1,-1,-1,-1},

/* Nokia Smart Messaging (short version) UDH
 * 1 byte  0x05      : IEI application port addressing scheme, 16 bit address
 * 1 byte  0x04      : IEI length
 * 2 bytes           : destination address : high & low byte
 * 2 bytes 0x00 0x00 : originator address  : high & low byte 
 */
{SMS_UDH_NokiaRingtone,         0x06,"\x05\x04\x15\x81\x00\x00",-1,-1,-1,-1},
{SMS_UDH_NokiaOperatorLogo,     0x06,"\x05\x04\x15\x82\x00\x00",-1,-1,-1,-1},
{SMS_UDH_NokiaCallerLogo,       0x06,"\x05\x04\x15\x83\x00\x00",-1,-1,-1,-1},
{SMS_UDH_NokiaWAP,		0x06,"\x05\x04\xc3\x4f\x00\x00",-1,-1,-1,-1},

{SMS_UDH_Linked_Long, 		0x06,"\x08\x04\x00\x00\x00\x00",-1,2,5,4},

{SMS_UDH_Linked_Long, 0,"",-1,-1,-1,-1}
};

typedef basic_string<GSM_SMS_UDH_Types> UDHList;

typedef enum {
	/* Unicode */
	SMS_Coding_Unicode_No_Compression = 1,
	SMS_Coding_Unicode_Compression,
	/* Default GSM alphabet */
	SMS_Coding_Default_No_Compression,
	SMS_Coding_Default_Compression,
	/* 8-bit */
	SMS_Coding_8bit
} SMS_Coding_Type;

typedef enum {
	SMS_Deliver = 1,
	SMS_Submit,
	SMS_Report
} SMS_Type;

/* SMS/SMSC format. Subset of values defined in GSM 03.40 section 9.2.3.9 */
typedef enum {
	SMS_Email = 1,
	SMS_Fax,
	SMS_Pager,
	SMS_Text
} SMS_Format;

/* SMS/SMSC relative validity.
 * Subset of values defined in GSM 03.40 section 9.2.3.12.1 +
 * additional Nokia value Max_Time */
const unsigned char SMS_RelativeValidity_1_Hour   = 0x0B;
const unsigned char SMS_RelativeValidity_6_Hours  = 0x47;
const unsigned char SMS_RelativeValidity_1_Day    = 0xA7;
const unsigned char SMS_RelativeValidity_3_Days   = 0xA9;
const unsigned char SMS_RelativeValidity_1_Week   = 0xAD;
const unsigned char SMS_RelativeValidity_Max_Time = 0xFF;

typedef enum {
	SMS_Status_Delivered = 1,
	SMS_Status_Pending,
	SMS_Status_Failed,
	SMS_Status_Unknown
} SMS_Report_Status;

typedef enum {
	SMS_Read = 1,
	SMS_UnRead,
	SMS_Sent,
	SMS_UnSent
} SMS_Icon;

class GSM_SMSNumbersSubEntry
{
public:
	GSM_SMSNumbersSubEntry();
	~GSM_SMSNumbersSubEntry();

	/** length (8 bits) + type (8 bits) + number */
	unsignedstring		PhoneNumber;

	GSM_Error		GetPhoneNumber(wchar_t *Destination);
	void			SetPhoneNumber(wchar_t *Text);
	void			SetNext(GSM_SMSNumbersSubEntry *Nxt);
	GSM_SMSNumbersSubEntry	*GetNext();
private:
	GSM_SMSNumbersSubEntry	*Next;
};

class GSM_SMSNumbers
{
public:
	GSM_SMSNumbers();
	~GSM_SMSNumbers();

	BOOLEAN 		GetNext(GSM_SMSNumbersSubEntry **En);
	GSM_Error	 	Add(wchar_t *Text);
	GSM_Error	 	Add(const unsigned char *Buffer, const int Len);
	void 			ClearAll();
private:
	GSM_SMSNumbersSubEntry	*Entries;
};

class GSM_SMSEntry
{
public:
	GSM_SMSEntry();
	~GSM_SMSEntry();

	//---------------------------------------------------------------------
	//Here start parameters with values set according to GSM specs
	//You can change them directly or using functions below
	//---------------------------------------------------------------------
	/**
	 * In SMS-Deliver and SMS-Submit TP-User-Data. GSM 03.40 section 9.2.3.24.
	 * In SMS-Status-Report not available
	 */
	unsignedstring		UserData;
	/**
	 * - In SMS-Deliver:       TP-Originating-Address. GSM 03.40 section 9.2.3.7.
	 * - In SMS-Submit:        TP-Destination-Address. GSM 03.40 section 9.2.3.8.
	 * - In SMS-Status-Report: TP-Recipient-Address.   GSM 03.40 section 9.2.3.14.
	 */
	/**
	 */
	/** length (8 bits) + type (8 bits) + number */
	unsignedstring		SMSCNumber;
	/**
	 * TP-Data-Coding-Scheme. GSM 03.40 section 9.2.3.10.
	 * Contains alphabet type, SMS class (and some others)
	 */
	unsigned char 		TPDCS;
	/**
	 * In SMS-Deliver      : not available
	 * In SMS-Submit       : TP-Validity-Period. GSM 03.40 section 9.2.3.12.
	 * In SMS-Status-Report: TP-Discharge Time.  GSM 03.40 section 9.2.3.13.
	 */
	unsigned char 		DateTime[7];
	/**
	 * In SMS-Deliver and SMS-Submit : not available
	 * In SMS-Status-Report : TP-Service-Centre-Time-Stamp. GSM 03.40 section 9.2.3.11.
	 */
	unsigned char 		SMSCTime[7];
	/**
	 * TP-Status in SMS-Status-Report. GSM 03.40 section 9.2.3.15.
	 */
	unsigned char		TPStatus;
	/**
	 * TP-User-Data-Length. GSM 03.40 section 9.2.3.16.
	 */
	unsigned char 		TPUDL;
	/**
	 * TP-Validity Period in SMS-Submit. GSM 03.40 section 9.2.3.12.
	 */
	unsigned char		TPVP;
	/**
	 * Byte contains in SMS-Deliver:
	 * - TP-Message-Type-Indicator     (2 bits) GSM 03.40 section 9.2.3.1
	 * - TP-More-Messages-To-Send      (1 bit). GSM 03.40 section 9.2.3.2
	 * - TP-Reply-Path                 (1 bit). GSM 03.40 section 9.2.3.17
	 * - TP-User-Data-Header-Indicator (1 bit). GSM 03.40 section 9.2.3.23
	 * - TP-Status-Report-Indicator    (1 bit). GSM 03.40 section 9.2.3.4
	 *
	 * Byte contains in SMS-Submit:
	 * - TP-Message-Type-Indicator     (2 bits) GSM 03.40 section 9.2.3.1
	 * - TP-Reject-Duplicates          (1 bit). GSM 03.40 section
	 * - TP-Validity-Period-Format     (2 bits).GSM 03.40 section 9.2.3.3
	 * - TP-Reply-Path                 (1 bit). GSM 03.40 section 9.2.3.17
	 * - TP-User-Data-Header-Indicator (1 bit). GSM 03.40 section 9.2.3.23
	 * - TP-Status-Report-Request      (1 bit). GSM 03.40 section 9.2.3.5
	 */
	unsigned char 		firstbyte;
	/**
	 * TP-Message Reference in SMS-Submit. GSM 03.40 section 9.2.3.6
	 */
	unsigned char 		TPMR;
	/**
	 * TP-Protocol-Identifier. GSM 03.40 section 9.2.3.9
	 */
	unsigned char 		TPPID;

	//---------------------------------------------------------------------
	//Functions for changing or decoding parameters declared above
	//---------------------------------------------------------------------
	GSM_Error		GetDateTime(GSM_DateTime *DT);
	GSM_Error		SetDateTime(GSM_DateTime *DT);
	GSM_Error		GetSMSCTime(GSM_DateTime *DT);
	GSM_Error		SetSMSCTime(GSM_DateTime *DT);
	SMS_Type		GetType();
	GSM_Error		SetType(SMS_Type Type);
	int 			GetClass();
	GSM_Error 		SetClass(int Class);
	SMS_Coding_Type		GetCoding();
	GSM_Error		SetCoding(SMS_Coding_Type Coding);
	int 			GetRelativeValidity();
	GSM_Error 		SetRelativeValidity(unsigned char Val);
	GSM_Error 		GetDecodedText(wchar_t *text);
	GSM_Error		SetDecodedText(wchar_t *Text);
	GSM_Error		GetText(unsignedstring *Text);
	GSM_Error		SetText(unsignedstring *Text, int l);
	GSM_Error 		GetUDH(unsignedstring *UDH);
	GSM_Error		SetUDH(unsignedstring *UDH);
	GSM_Error		GetSMSCNumber(wchar_t *Destination);
	void			SetSMSCNumber(wchar_t *Text);
	SMS_Report_Status 	GetReportStatus();
	BOOLEAN			GetSenderSMSCReply();
	BOOLEAN			SetSenderSMSCReply(BOOLEAN x);
	BOOLEAN			GetReportRequest();
	BOOLEAN			SetReportRequest(BOOLEAN x);
	GSM_Error		DecodeUDH(UDHList *List);
	BOOLEAN 		SetReplace(int i);
	BOOLEAN 		GetReplace(int *i);
	BOOLEAN 		SetRejectDuplicates(BOOLEAN enable);

	//---------------------------------------------------------------------
	//one SMS can contain max. one phone number. We allow more because of
	//solutions proposed in Nokia Series 40 3.0 phones
	GSM_SMSNumbers		PhoneNumbers;
	//---------------------------------------------------------------------

	//---------------------------------------------------------------------
	//part of parameters connected with SMS in phone
	//---------------------------------------------------------------------
	GSM_MemoryType		Memory;
	wchart			ID;
	wchart			Name;
	SMS_Icon		Icon;
	BOOLEAN			SaveDateTimeAvailable;
	GSM_DateTime		SaveDateTime;

	//other
	unsigned char		MakeSMSIDFromTime(void);
private:
	GSM_Error 		GetSMSDateTime(GSM_DateTime *DT, const unsigned char *buffer);
	GSM_Error 		SetSMSDateTime(GSM_DateTime *DT, unsigned char *buffer);
};

class GSM_SMSListSubEntry
{
public:
	GSM_SMSListSubEntry();
	~GSM_SMSListSubEntry();

	void			SetSMS(GSM_SMSEntry *SMS2);
	GSM_SMSEntry 		*GetSMS();
	void			SetNext(GSM_SMSListSubEntry *Nxt);
	GSM_SMSListSubEntry	*GetNext();
private:
	GSM_SMSListSubEntry	*Next;
	GSM_SMSEntry		SMS;
};

class GSM_SMSList
{
public:
	GSM_SMSList();
	~GSM_SMSList();

	//---------------------------------------------------------------------
	//part of parameters connected with SMS in phone
	//---------------------------------------------------------------------
	int 			Folder; //number

	BOOLEAN 		GetNext(GSM_SMSListSubEntry **En);
	GSM_Error	 	Add(GSM_SMSEntry *SMS);
	void 			ClearAll();
private:
	GSM_SMSListSubEntry	*Entries;
};

typedef struct {
	int 			SMSTemplates;

	int 			SMSFree;
	int			SMSUnRead;
	int			SMSRead;

	GSM_MemoryType		Memory;
} GSM_SMSStatus;
					  
class GSM_SMSC
{
public:
	GSM_SMSC();
	~GSM_SMSC();

	int			Location;
	unsigned char		RelativeValidity;
	SMS_Format		Format;
	
	void			SetName(wchar_t *N);
	wchar_t 		*GetName();
	void			SetSMSCNumber(wchar_t *N);
	wchar_t 		*GetSMSCNumber();
	void			SetDefaultNumber(wchar_t *N);
	wchar_t 		*GetDefaultNumber();
private:
	wchar_t			*SMSCNumber;
	wchar_t			*DefaultNumber;
	wchar_t			*Name;
};

#endif
