/* (C) 2005 by Marcin Wiacek www.mwiacek.com */

#include "gsmsms.h"
#include "../gsmmisc.h"
#include "../../misc/coding/coding.h"

GSM_Error GSM_DecodeSMSNumber(wchar_t *Destination, const unsigned char *Source, int len, BOOLEAN semioctets)
{
	int len2 = Source[0],pos=0;

	if (len==0) {
		Destination[0]=0;
		return GSM_Return_Error(GSM_ERR_NONE);
	}
	
	if (semioctets) {
		if (len2 % 2) len2++;
		len2 = len2/2+1;
	}
	len2--;

	if ((Source[1] & 112) == (0xD0 & 112)) {		
		//alphanumeric
		len2++;
//		GSM_DecodeSMSText(Destination, Source+2, Source[0], Source[0], 0);
		GSM_DecodeSMSText(Destination, Source+2, len2, len2, 0);
		return GSM_Return_Error(GSM_ERR_NONE);
	}
	if ((Source[1] & 112) == (0x91 & 112)) Destination[pos++] = '+';
	BCDToUnicode(Source+2,Destination+pos,len2);
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_EncodeSMSNumber(unsignedstring *Destination, wchar_t *Source, BOOLEAN semioctets)
{
        unsigned char 	buff[50],type=0x91;
	int 		len2,i,len3,len4,used;
	GSM_Error	error;

	if (UnicodeLength(Source) == 0) return GSM_Return_Error(GSM_ERR_NONE);
	for (i=0;i<UnicodeLength(Source);i++) {
		if (i==0 && Source[0] == '+') continue;
		if (!isdigit(UnicodeToStringReturn(Source)[i])) {
			error = GSM_EncodeSMSText(buff, Source, &len4, &len2, 50, &used, 0);
			if (error.Code != GSM_ERR_NONE) return error;
			len3 = used;
			type = 0xD0;
			break;
		}
	}

	if (type == 0x91) {
		if (Source[0] == '+') {
			len2 = StringToBCD((const unsigned char *)UnicodeToStringReturn(Source+1), buff, UnicodeLength(Source)-1);
			len3 = UnicodeLength(Source)-1;
		} else {
			len2 = StringToBCD((const unsigned char *)UnicodeToStringReturn(Source), buff, UnicodeLength(Source));
			len3 = UnicodeLength(Source);
			type = 0x81;
		}
	}

	if (!semioctets) len3=len2+1;

	Destination->push_back(len3);
	Destination->push_back(type);
	Destination->append(buff,len2);
	return GSM_Return_Error(GSM_ERR_NONE);
}

/* ETSI GSM 03.38, section 6.2.1: Default alphabet for SMS messages */
static unsigned char GSM_DefaultAlphabetUnicode[128+1][2] =
{
	{0x00,0x40},{0x00,0xa3},{0x00,0x24},{0x00,0xA5},
	{0x00,0xE8},{0x00,0xE9},{0x00,0xF9},{0x00,0xEC},/*0x08*/
	{0x00,0xF2},{0x00,0xC7},{0x00,'\n'},{0x00,0xD8},
	{0x00,0xF8},{0x00,'\r'},{0x00,0xC5},{0x00,0xE5},
	{0x03,0x94},{0x00,0x5f},{0x03,0xA6},{0x03,0x93},
	{0x03,0x9B},{0x03,0xA9},{0x03,0xA0},{0x03,0xA8},
	{0x03,0xA3},{0x03,0x98},{0x03,0x9E},{0x00,0xb9},
	{0x00,0xC6},{0x00,0xE6},{0x00,0xDF},{0x00,0xC9},/*0x20*/
	{0x00,' ' },{0x00,'!' },{0x00,'\"'},{0x00,'#' },
	{0x00,0xA4},{0x00,'%' },{0x00,'&' },{0x00,'\''},
	{0x00,'(' },{0x00,')' },{0x00,'*' },{0x00,'+' },
	{0x00,',' },{0x00,'-' },{0x00,'.' },{0x00,'/' },/*0x30*/
	{0x00,'0' },{0x00,'1' },{0x00,'2' },{0x00,'3' },
	{0x00,'4' },{0x00,'5' },{0x00,'6' },{0x00,'7' },
	{0x00,'8' },{0x00,'9' },{0x00,':' },{0x00,';' },
	{0x00,'<' },{0x00,'=' },{0x00,'>' },{0x00,'?' },/*0x40*/
	{0x00,0xA1},{0x00,'A' },{0x00,'B' },{0x00,'C' },
	{0x00,'D' },{0x00,'E' },{0x00,'F' },{0x00,'G' },
	{0x00,'H' },{0x00,'I' },{0x00,'J' },{0x00,'K' },
	{0x00,'L' },{0x00,'M' },{0x00,'N' },{0x00,'O' },
	{0x00,'P' },{0x00,'Q' },{0x00,'R' },{0x00,'S' },
	{0x00,'T' },{0x00,'U' },{0x00,'V' },{0x00,'W' },
	{0x00,'X' },{0x00,'Y' },{0x00,'Z' },{0x00,0xC4},
	{0x00,0xD6},{0x00,0xD1},{0x00,0xDC},{0x00,0xA7},
	{0x00,0xBF},{0x00,'a' },{0x00,'b' },{0x00,'c' },
	{0x00,'d' },{0x00,'e' },{0x00,'f' },{0x00,'g' },
	{0x00,'h' },{0x00,'i' },{0x00,'j' },{0x00,'k' },
	{0x00,'l' },{0x00,'m' },{0x00,'n' },{0x00,'o' },
	{0x00,'p' },{0x00,'q' },{0x00,'r' },{0x00,'s' },
	{0x00,'t' },{0x00,'u' },{0x00,'v' },{0x00,'w' },
	{0x00,'x' },{0x00,'y' },{0x00,'z' },{0x00,0xE4},
	{0x00,0xF6},{0x00,0xF1},{0x00,0xFC},{0x00,0xE0},
	{0x00,0x00}
};

/* ETSI GSM 3.38
 * Some sequences of 2 default alphabet chars (for example,
 * 0x1b, 0x65) are visible as one single additional char (for example,
 * 0x1b, 0x65 gives Euro char saved in Unicode as 0x20, 0xAC)
 * This table contains:
 * 1. two first chars means sequence of chars from GSM default alphabet
 * 2. two second is target (encoded) char saved in Unicode
 */
static unsigned char GSM_DefaultAlphabetCharsExtension[][4] =
{
	{0x1b,0x14,0x00,0x5e},	/* ^	*/
	{0x1b,0x28,0x00,0x7b},	/* {	*/
	{0x1b,0x29,0x00,0x7d},	/* }	*/
	{0x1b,0x2f,0x00,0x5c},	/* \	*/
	{0x1b,0x3c,0x00,0x5b},	/* [	*/
	{0x1b,0x3d,0x00,0x7E},	/* ~	*/
	{0x1b,0x3e,0x00,0x5d},	/* ]	*/
	{0x1b,0x40,0x00,0x7C},	/* |	*/
	{0x1b,0x65,0x20,0xAC},	/* Euro */
	{0x00,0x00,0x00,0x00}
};

GSM_Error GSM_DecodeSMSText(wchar_t *Destination, const unsigned char *Source, int len, int len3, int udhlen)
{
	unsigned int		i,chr,j,z,p,q,r,len2;
	wchar_t			prev;

	i = 0;
	do {
		i+=7;
	} while ((i-udhlen)%i<0);
	len2 = len-(udhlen*8 + (i-udhlen)%i) / 7;
//  	len2 = len;
	j = 1;
	for (i=1;i<(unsigned int)udhlen*7;i++) {
		if (j == 128) {
			j = 1;
		} else {
			j = j * 2;
		}
	}
	i = 0;
	q = 0;
	for (p=0;p<len2;p++) {
		chr = 0;
		for (z=1;z<128;z=z*2) {
			if (Source[i]&j) chr += z;
			if (j == 128) {
				j = 1;
				i++;
				if (i <= (unsigned int)len3) continue;
				return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
			}
			j = j * 2;
		}
		Destination[q] = GSM_DefaultAlphabetUnicode[chr][0]*256+GSM_DefaultAlphabetUnicode[chr][1];
		if (p>0) {
			r = 0;
			while (GSM_DefaultAlphabetCharsExtension[r][0] != 0x00) {
				if (prev == GSM_DefaultAlphabetCharsExtension[r][0] &&
				    chr == GSM_DefaultAlphabetCharsExtension[r][1]) {
					q--;
					Destination[q] = GSM_DefaultAlphabetCharsExtension[r][2]*256+
						  GSM_DefaultAlphabetCharsExtension[r][3];
					break;
				}
				r++;
			}
		}
		prev = chr;
		q++;
	}
	Destination[q] = 0;

	return GSM_Return_Error(GSM_ERR_NONE);
}

/* There are many national chars with "adds". In phone they're normally
 * changed to "plain" Latin chars. We have such functionality too.
 * This table is automatically created from convert.txt file (see
 * /docs/developers) using --makeconverttable. It contains such chars
 * to replace in order:
 * 1. original char (Unicode) 2. destination char (Unicode)
 */
static unsigned char ConvertTable[] =
"\x00\xc0\x00\x41\x00\xe0\x00\x61\x00\xc1\x00\x41\x00\xe1\x00\x61\x00\xc2\x00\x41\x00\xe2\x00\x61\x00\xc3\x00\x41\x00\xe3\x00\x61\x1e\xa0\x00\x41\x1e\xa1\x00\x61\x1e\xa2\x00\x41\x1e\xa3\x00\x61\x1e\xa4\x00\x41\x1e\xa5\x00\x61\x1e\xa6\x00\x41\x1e\xa7\x00\x61\x1e\xa8\x00\x41\x1e\xa9\x00\x61\x1e\xaa\x00\x41\x1e\xab\x00\x61\x1e\xac\x00\x41\x1e\xad\x00\x61\x1e\xae\x00\x41\x1e\xaf\x00\x61\x1e\xb0\x00\x41\x1e\xb1\x00\x61\x1e\xb2\x00\x41\x1e\xb3\x00\x61\x1e\xb4\x00\x41\x1e\xb5\x00\x61\x1e\xb6\x00\x41\x1e\xb7\x00\x61\x01\xcd\x00\x41\x01\xce\x00\x61\x01\x00\x00\x41\x01\x01\x00\x61\x01\x02\x00\x41\x01\x03\x00\x61\x01\x04\x00\x41\x01\x05\x00\x61\x01\xfb\x00\x61\x01\x06\x00\x43\x01\x07\x00\x63\x01\x08\x00\x43\x01\x09\x00\x63\x01\x0a\x00\x43\x01\x0b\x00\x63\x01\x0c\x00\x43\x01\x0d\x00\x63\x00\xe7"\
"\x00\x63\x01\x0e\x00\x44\x01\x0f\x00\x64\x01\x10\x00\x44\x01\x11\x00\x64\x00\xc8\x00\x45\x00\xca\x00\x45\x00\xea\x00\x65\x00\xcb\x00\x45\x00\xeb\x00\x65\x1e\xb8\x00\x45\x1e\xb9\x00\x65\x1e\xba\x00\x45\x1e\xbb\x00\x65\x1e\xbc\x00\x45\x1e\xbd\x00\x65\x1e\xbe\x00\x45\x1e\xbf\x00\x65\x1e\xc0\x00\x45\x1e\xc1\x00\x65\x1e\xc2\x00\x45\x1e\xc3\x00\x65\x1e\xc4\x00\x45\x1e\xc5\x00\x65\x1e\xc6\x00\x45\x1e\xc7\x00\x65\x01\x12\x00\x45\x01\x13\x00\x65\x01\x14\x00\x45\x01\x15\x00\x65\x01\x16\x00\x45\x01\x17\x00\x65\x01\x18\x00\x45\x01\x19\x00\x65\x01\x1a\x00\x45\x01\x1b\x00\x65\x01\x1c\x00\x47\x01\x1d\x00\x67\x01\x1e\x00\x47\x01\x1f\x00\x67\x01\x20\x00\x47\x01\x21\x00\x67\x01\x22\x00\x47\x01\x23\x00\x67\x01\x24\x00\x48\x01\x25\x00\x68\x01\x26\x00\x48\x01\x27\x00\x68\x00\xcc\x00\x49\x00\xcd\x00\x49\x00\xed"\
"\x00\x69\x00\xce\x00\x49\x00\xee\x00\x69\x00\xcf\x00\x49\x00\xef\x00\x69\x01\x28\x00\x49\x01\x29\x00\x69\x01\x2a\x00\x49\x01\x2b\x00\x69\x01\x2c\x00\x49\x01\x2d\x00\x69\x01\x2e\x00\x49\x01\x2f\x00\x69\x01\x30\x00\x49\x01\x31\x00\x69\x01\xcf\x00\x49\x01\xd0\x00\x69\x1e\xc8\x00\x49\x1e\xc9\x00\x69\x1e\xca\x00\x49\x1e\xcb\x00\x69\x01\x34\x00\x4a\x01\x35\x00\x6a\x01\x36\x00\x4b\x01\x37\x00\x6b\x01\x39\x00\x4c\x01\x3a\x00\x6c\x01\x3b\x00\x4c\x01\x3c\x00\x6c\x01\x3d\x00\x4c\x01\x3e\x00\x6c\x01\x3f\x00\x4c\x01\x40\x00\x6c\x01\x41\x00\x4c\x01\x42\x00\x6c\x01\x43\x00\x4e\x01\x44\x00\x6e\x01\x45\x00\x4e\x01\x46\x00\x6e\x01\x47\x00\x4e\x01\x48\x00\x6e\x01\x49\x00\x6e\x00\xd2\x00\x4f\x00\xd3\x00\x4f\x00\xf3\x00\x6f\x00\xd4\x00\x4f\x00\xf4\x00\x6f\x00\xd5\x00\x4f\x00\xf5\x00\x6f\x01\x4c\x00\x4f\x01\x4d"\
"\x00\x6f\x01\x4e\x00\x4f\x01\x4f\x00\x6f\x01\x50\x00\x4f\x01\x51\x00\x6f\x01\xa0\x00\x4f\x01\xa1\x00\x6f\x01\xd1\x00\x4f\x01\xd2\x00\x6f\x1e\xcc\x00\x4f\x1e\xcd\x00\x6f\x1e\xce\x00\x4f\x1e\xcf\x00\x6f\x1e\xd0\x00\x4f\x1e\xd1\x00\x6f\x1e\xd2\x00\x4f\x1e\xd3\x00\x6f\x1e\xd4\x00\x4f\x1e\xd5\x00\x6f\x1e\xd6\x00\x4f\x1e\xd7\x00\x6f\x1e\xd8\x00\x4f\x1e\xd9\x00\x6f\x1e\xda\x00\x4f\x1e\xdb\x00\x6f\x1e\xdc\x00\x4f\x1e\xdd\x00\x6f\x1e\xde\x00\x4f\x1e\xdf\x00\x6f\x1e\xe0\x00\x4f\x1e\xe1\x00\x6f\x1e\xe2\x00\x4f\x1e\xe3\x00\x6f\x01\x54\x00\x52\x01\x55\x00\x72\x01\x56\x00\x52\x01\x57\x00\x72\x01\x58\x00\x52\x01\x59\x00\x72\x01\x5a\x00\x53\x01\x5b\x00\x73\x01\x5c\x00\x53\x01\x5d\x00\x73\x01\x5e\x00\x53\x01\x5f\x00\x73\x01\x60\x00\x53\x01\x61\x00\x73\x01\x62\x00\x54\x01\x63\x00\x74\x01\x64\x00\x54\x01\x65"\
"\x00\x74\x01\x66\x00\x54\x01\x67\x00\x74\x00\xd9\x00\x55\x00\xda\x00\x55\x00\xfa\x00\x75\x00\xdb\x00\x55\x00\xfb\x00\x75\x01\x68\x00\x55\x01\x69\x00\x75\x01\x6a\x00\x55\x01\x6b\x00\x75\x01\x6c\x00\x55\x01\x6d\x00\x75\x01\x6e\x00\x55\x01\x6f\x00\x75\x01\x70\x00\x55\x01\x71\x00\x75\x01\x72\x00\x55\x01\x73\x00\x75\x01\xaf\x00\x55\x01\xb0\x00\x75\x01\xd3\x00\x55\x01\xd4\x00\x75\x01\xd5\x00\x55\x01\xd6\x00\x75\x01\xd7\x00\x55\x01\xd8\x00\x75\x01\xd9\x00\x55\x01\xda\x00\x75\x01\xdb\x00\x55\x01\xdc\x00\x75\x1e\xe4\x00\x55\x1e\xe5\x00\x75\x1e\xe6\x00\x55\x1e\xe7\x00\x75\x1e\xe8\x00\x55\x1e\xe9\x00\x75\x1e\xea\x00\x55\x1e\xeb\x00\x75\x1e\xec\x00\x55\x1e\xed\x00\x75\x1e\xee\x00\x55\x1e\xef\x00\x75\x1e\xf0\x00\x55\x1e\xf1\x00\x75\x01\x74\x00\x57\x01\x75\x00\x77\x1e\x80\x00\x57\x1e\x81\x00\x77\x1e\x82"\
"\x00\x57\x1e\x83\x00\x77\x1e\x84\x00\x57\x1e\x85\x00\x77\x00\xdd\x00\x59\x00\xfd\x00\x79\x00\xff\x00\x79\x01\x76\x00\x59\x01\x77\x00\x79\x01\x78\x00\x59\x1e\xf2\x00\x59\x1e\xf3\x00\x75\x1e\xf4\x00\x59\x1e\xf5\x00\x79\x1e\xf6\x00\x59\x1e\xf7\x00\x79\x1e\xf8\x00\x59\x1e\xf9\x00\x79\x01\x79\x00\x5a\x01\x7a\x00\x7a\x01\x7b\x00\x5a\x01\x7c\x00\x7a\x01\x7d\x00\x5a\x01\x7e\x00\x7a\x01\xfc\x00\xc6\x01\xfd\x00\xe6\x01\xfe\x00\xd8\x01\xff\x00\xf8\x00\x00";

GSM_Error GSM_EncodeSMSText(unsigned char *Destination, const wchar_t *Source, int *GetFromSRC, int *SavedInSMS, int MaxInSMS, int *UsedInSMS, int udhlen)
{
	unsigned int	pos=0,j,destbit,z,gsmcharlen,l=0;
	unsigned char 	gsmchar[2];
	BOOLEAN 	found;
	wchar_t 	srcchar;

	(*SavedInSMS) 		= 0;
	(*GetFromSRC) = 0;
	(*UsedInSMS) = 0;
	Destination[l] 	= 0;
	destbit		= 1;
	for (j=1;j<udhlen*7;j++) {
		if (destbit==128) {
			destbit = 1;
			continue;
		}
		destbit*=2;
	}
	while (Source[pos] != 0) {
		found 		= FALSE;
		gsmcharlen 	= 1;

		j = 0;
		while (GSM_DefaultAlphabetCharsExtension[j][0]*256+GSM_DefaultAlphabetCharsExtension[j][1]!=0) {
			if (GSM_DefaultAlphabetCharsExtension[j][2]*256+GSM_DefaultAlphabetCharsExtension[j][3] == Source[pos]) {
				gsmchar[0] = GSM_DefaultAlphabetCharsExtension[j][0];
				gsmchar[1] = GSM_DefaultAlphabetCharsExtension[j][1];
				gsmcharlen = 2;
				found = TRUE;
				break;
			}
			j++;
		}

		if (found==FALSE) {
			srcchar = Source[pos];
			j = 0;
			while (ConvertTable[j*4]*256+ConvertTable[j*4+1]!=0) {
				if (Source[pos] == ConvertTable[j*4]*256+ConvertTable[j*4+1]) {
					srcchar = ConvertTable[j*4+2]*256+ConvertTable[j*4+3];
					break;
				}
				j++;
			}

			j = 0;
			while (GSM_DefaultAlphabetUnicode[j][0]*256+GSM_DefaultAlphabetUnicode[j][1]!=0) {
				if (GSM_DefaultAlphabetUnicode[j][0]*256+GSM_DefaultAlphabetUnicode[j][1] == Source[pos]) {
					gsmchar[0] = j;
					found = TRUE;
					break;
				}
				if (GSM_DefaultAlphabetUnicode[j][0]*256+GSM_DefaultAlphabetUnicode[j][1] == srcchar) {
					gsmchar[0] = j;
					found = TRUE;
					break;
				}
				j++;
			}
		}

		if (found==FALSE) gsmchar[0] = '?';

		for (j=0;j<gsmcharlen;j++) {
			for (z=1;z<128;z*=2) {
				if (destbit==256) {
					l++;
					if (l >= MaxInSMS) {
						return GSM_Return_Error(GSM_ERR_NONE);
					}
					Destination[l] = 0;
					destbit = 1;
				}
				if (gsmchar[j] & z) Destination[l] += destbit;
				destbit*=2;
			}
		}
		(*GetFromSRC)++;
		(*SavedInSMS)+=gsmcharlen;
		(*UsedInSMS) = l+1;
		pos++;
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

//-----------------------------------------------------------------------------

unsigned char GSM_SMSEntry::MakeSMSIDFromTime(void)
{
	GSM_DateTime 	Date;
	unsigned char	retval;

	GSM_GetCurrentDateTime (&Date);
	retval = Date.Second;
	switch (Date.Minute/10) {
		case 2: case 7: 	retval = retval +  60; break;
		case 4: case 8: 	retval = retval + 120; break;
		case 9: case 5: case 0: retval = retval + 180; break;
	}
	retval += Date.Minute/10;
	return retval;
}

GSM_Error GSM_SMSEntry::DecodeUDH(UDHList *List)
{
        int 			i=0,j,pos=1;
	unsignedstring 		UDH;
	GSM_Error 		error;
	BOOLEAN			wrong;
	GSM_SMS_UDH_Types	New;

	List->clear();

	error = GetUDH(&UDH);
	if (error.Code != GSM_ERR_NONE) return error;

	while (pos!=UDH.length()) {
		while(UDHHeaders[i].Length!=0) {
			wrong = FALSE;
			if ((int)(UDH.length()-pos)>=UDHHeaders[i].Length) {
				for (j=0;j<UDHHeaders[i].Length;j++) {
					if (UDHHeaders[i].ID8bit!=-1 && j==UDHHeaders[i].ID8bit) continue;
					if (UDHHeaders[i].PartNumber8bit!=-1 && j==UDHHeaders[i].PartNumber8bit) continue;
					if (UDHHeaders[i].AllParts8bit!=-1 && j==UDHHeaders[i].AllParts8bit) continue;
					if (UDHHeaders[i].ID16bit!=-1 && (j==UDHHeaders[i].ID16bit || j==UDHHeaders[i].ID16bit+1)) continue;
					if (UDHHeaders[i].Text[j]==UDH.data()[j+pos]) continue;
					wrong = TRUE;
					break;
				}
			} else {
				wrong = TRUE;
			}
			if (!wrong) {
				memcpy(&New,&UDHHeaders[i],sizeof(GSM_SMS_UDH_Types));
				memcpy(New.Text,UDH.data()+pos,UDHHeaders[i].Length);
				if (New.ID8bit!=-1) New.ID8bit=New.Text[New.ID8bit];
				if (New.ID16bit!=-1) New.ID16bit=New.Text[New.ID16bit]*256+New.Text[New.ID16bit+1];
				if (New.AllParts8bit!=-1) New.AllParts8bit=New.Text[New.AllParts8bit];
				if (New.PartNumber8bit!=-1) New.PartNumber8bit=New.Text[New.PartNumber8bit];
				List->push_back(New);
				pos+=UDHHeaders[i].Length;
				break;
			}
			i++;
		}
		if (!wrong) continue;
		return GSM_Return_Error(GSM_ERR_NOT_SUPPORTED);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_SMSEntry::GSM_SMSEntry() 
{
	TPDCS 	  = 0;
	memset(DateTime,0,sizeof(DateTime));
	memset(SMSCTime,0,sizeof(SMSCTime));
	TPStatus  = 0;
	TPUDL	  = 0;
	TPVP 	  = 0;
	firstbyte = 0;
	TPMR 	  = 0;
	TPPID 	  = 0;

	Icon = SMS_UnRead;
	SaveDateTimeAvailable = FALSE;


	SetCoding(SMS_Coding_Default_No_Compression);
}

GSM_SMSEntry::~GSM_SMSEntry() 
{
}

SMS_Type GSM_SMSEntry::GetType()
{
	switch (firstbyte & 0x03) {
		case  1: return SMS_Submit;
		case  2: return SMS_Report;
		default: return SMS_Deliver;
	}
	return SMS_Deliver;
}

/* GSM 03.40 section 9.2.3.1 (TP-Message-Type-Indicator) */
GSM_Error GSM_SMSEntry::SetType(SMS_Type Type)
{
	/* bits 0 and 1 */
	firstbyte &= 0xFC;
	switch (Type) {
		case SMS_Deliver: 		  break;
		case SMS_Submit : firstbyte += 1; break;
		case SMS_Report : firstbyte += 2; break;
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

//to test
SMS_Report_Status GSM_SMSEntry::GetReportStatus()
{
	/* GSM 03.40 section 9.2.3.15 (TP-ST) */	
	if ((TPStatus & 0x80) == 0x80) return SMS_Status_Unknown;

	if ((TPStatus & 0x7F) <= 2) return SMS_Status_Delivered;
	if ((TPStatus & 0x7F) >= 0x20 &&
	    (TPStatus & 0x7F) <= 0x3F) return SMS_Status_Pending;
	if ((TPStatus & 0x7F) >= 0x40) return SMS_Status_Failed;
	return SMS_Status_Unknown;
}

SMS_Coding_Type GSM_SMSEntry::GetCoding()
{
	if (GetType() == SMS_Report) return SMS_Coding_8bit;

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xC0) == 0) {
		/* bits 7..4 set to 00xx */
		if ((TPDCS & 0xC) == 0xC) {
			//reserved alphabet value in TPDCS
		} else {
			if (TPDCS == 0) 	    return SMS_Coding_Default_No_Compression;
			if ((TPDCS & 0x2C) == 0x00) return SMS_Coding_Default_No_Compression;
			if ((TPDCS & 0x2C) == 0x20) return SMS_Coding_Default_Compression;
			if ((TPDCS & 0x2C) == 0x08) return SMS_Coding_Unicode_No_Compression;
			if ((TPDCS & 0x2C) == 0x28) return SMS_Coding_Unicode_Compression;
		}
	} else if ((TPDCS & 0xF0) >= 0x40 && (TPDCS & 0xF0) <= 0xB0) {
		/* bits 7..4 set to 0100 ... 1011 */
		//reserved coding group in TPDCS
	} else if (((TPDCS & 0xF0) == 0xC0) || ((TPDCS & 0xF0) == 0xD0)) {
		/* bits 7..4 set to 1100 or 1101 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
		} else {
			return SMS_Coding_Default_No_Compression;
		}
	} else if ((TPDCS & 0xF0) == 0xE0) {
		/* bits 7..4 set to 1110 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
		} else {
			return SMS_Coding_Unicode_No_Compression;
		}
	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
		} else {
			if ((TPDCS & 4) == 0) return SMS_Coding_Default_No_Compression;
		}
	}	
	return SMS_Coding_8bit;
}

//to test
GSM_Error GSM_SMSEntry::SetCoding(SMS_Coding_Type Coding)
{
	if (GetType() == SMS_Report) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xC0) == 0) {
		/* bits 7..4 set to 00xx */
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			TPDCS &= 0xD3; 	// set bits 2 and 5 to 0
			TPDCS += 8; 	// set bit 3 to 1
			break;
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			TPDCS &= 0xD3; 	// set bits 2, 3 and 5 to 0
			break;
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			TPDCS &= 0xF3; 	// set bit 3 to 0
			TPDCS += 4; 	// set bit 3 to 1
			//return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		}
	} else if ((TPDCS & 0xF0) >= 0x40 && (TPDCS & 0xF0) <= 0xB0) {
		/* bits 7..4 set to 0100 ... 1011 */
		//reserved coding group in TPDCS
		return GSM_Return_Error(GSM_ERR_UNKNOWN);
	} else if (((TPDCS & 0xF0) == 0xC0) || ((TPDCS & 0xF0) == 0xD0)) {
		/* bits 7..4 set to 1100 or 1101 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		}
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			break;
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);			
		}
	} else if ((TPDCS & 0xF0) == 0xE0) {
		/* bits 7..4 set to 1110 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		}
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			break;
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);			
		}
    	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);	
		}
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);				
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			TPDCS &= 0xEF; 	// set bit 4 to 0
			break;
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			TPDCS &= 0xEF; 	// set bit 4 to 0
			TPDCS += 0x10;  // set bit 4 to 1
		}
	}	
	return GSM_Return_Error(GSM_ERR_NONE);
}

int GSM_SMSEntry::GetClass() 
{
	if (GetType() == SMS_Report) return -1;

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xD0) == 0x10) {
		/* bits 7..4 set to 00x1 */
		if ((TPDCS & 0xC) == 0xC) {
			//reserved alphabet value in TPDCS
		} else {
			return (TPDCS & 3);
		}
	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
		} else {
			return (TPDCS & 3);
		}
	}
	return -1;
}

//to test
GSM_Error GSM_SMSEntry::SetClass(int Class) 
{
	if (GetType() == SMS_Report) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	if (Class < -1 || Class > 3) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xC0) == 0) {
		/* bits 7..4 set to 00xx */
		if ((TPDCS & 0xC) == 0xC) {
			//reserved alphabet value in TPDCS
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		}
		TPDCS &= 0xEF; //set bit 4 to 0
		if (Class == -1) return GSM_Return_Error(GSM_ERR_NONE);
		TPDCS += 0x10; //set bit 4 to 1
    	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);	
		}
		if (Class == -1) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	} else {
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}
	TPDCS += Class;

	return GSM_Return_Error(GSM_ERR_NONE);
}

//todo
int GSM_SMSEntry::GetRelativeValidity() 
{
	return -1;
}

//todo
GSM_Error GSM_SMSEntry::SetRelativeValidity(unsigned char Val)
{
	/* GSM 03.40 section 9.2.3.3 (TP-Validity-Period-Format) */
	/* Bits 4 and 3: 10. TP-VP field present and integer represent (relative) */
	firstbyte |= 0x10;

	TPVP=Val;

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::GetDecodedText(wchar_t *text)
{
	unsignedstring  	TMP, UDH;
	unsigned int		i = 0;
	unsigned char 		*Buffer;
	GSM_Error		error;

	error = GetText(&TMP);
	if (error.Code != GSM_ERR_NONE) return error;

	Buffer=(unsigned char *)TMP.data();

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		if ((TMP.size() % 2) != 0) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		for (i=0;i<TMP.size()/2;i++) {
			text[i] = Buffer[i*2]*256 + Buffer[i*2+1];
		}
		text[i] = 0;
		return GSM_Return_Error(GSM_ERR_NONE);
	case SMS_Coding_Default_No_Compression:
		error = GetUDH(&UDH);
		GSM_DecodeSMSText(text, Buffer, TPUDL, TMP.size(), UDH.size());
		return GSM_Return_Error(GSM_ERR_NONE);
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
	default:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}
}

GSM_Error GSM_SMSEntry::SetDecodedText(wchar_t *Text)
{
	unsignedstring  TMP,TMP2;
	unsigned char	Buffer[200];
	int		i,j,x,used;
	GSM_Error	error;

	error = GetUDH(&TMP2);
	if (error.Code != GSM_ERR_NONE) return error;

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		//fixme
		for (i=0;i<UnicodeLength(Text);i++) {
			TMP.push_back(Text[i]/256);
			TMP.push_back(Text[i]%256);
		}
		return SetText(&TMP,UnicodeLength(Text)*2);
	case SMS_Coding_Default_No_Compression:
//		error = GSM_EncodeSMSText(Buffer, Text, &j, 133-TMP2.size(), TMP2.size());
		error = GSM_EncodeSMSText(Buffer, Text, &x, &j, 140-TMP2.size(), &used, TMP2.size());
		if (error.Code != GSM_ERR_NONE) return error;
		for (i=0;i<used;i++) TMP.push_back(Buffer[i]);
//printf("%i\n",(j*7/8));
		return SetText(&TMP,j);		
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	default:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}
}

GSM_Error GSM_SMSEntry::GetText(unsignedstring *Text) 
{
	unsignedstring 	TMP;
	GSM_Error	error;

	error = GetUDH(&TMP);
	if (error.Code != GSM_ERR_NONE) return error;

	Text->append((const unsigned char *)UserData.data()+TMP.size(),UserData.size()-TMP.size());

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::SetText(unsignedstring *Text, int l)
{
	unsignedstring 	TMP;
	int		Len;
	GSM_Error	error;

	error = GetUDH(&TMP);
	if (error.Code != GSM_ERR_NONE) return error;

	Len = TMP.size() + Text->size();
	if (Len > 140) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		if ((Text->size() % 2) != 0) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		TPUDL = l*2;
		break;
	case SMS_Coding_Default_No_Compression:
		TPUDL = l;
		break;
	case SMS_Coding_8bit:
		TPUDL = l;
		break;
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}

	UserData.clear();
	UserData.append((const unsigned char *)TMP.data(),TMP.size());
	UserData.append((const unsigned char *)Text->data(),Text->size());

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::GetDateTime(GSM_DateTime *DT)
{
	return GetSMSDateTime(DT,DateTime);
}

BOOLEAN	GSM_SMSEntry::GetSenderSMSCReply()
{
	if (GetType() != SMS_Submit) return FALSE;

	if ((firstbyte & 0x80)!=0) return TRUE;
	return FALSE;
}

/* GSM 03.40 section 9.2.3.17 (TP-Reply-Path) */
BOOLEAN GSM_SMSEntry::SetSenderSMSCReply(BOOLEAN x)
{
	if (GetType() != SMS_Submit) return FALSE;

	if (x==TRUE) {
		firstbyte |= 0x80;
	} else {
		firstbyte &= 0x7F;//clear bit
	}
	return TRUE;
}

BOOLEAN GSM_SMSEntry::GetReportRequest()
{
	if (GetType() != SMS_Submit) return FALSE;

	if ((firstbyte & 0x20)!=0) return TRUE;
	return FALSE;
}

/* GSM 03.40 section 9.2.3.5 (TP-Status-Raport-Request) */
BOOLEAN GSM_SMSEntry::SetReportRequest(BOOLEAN x)
{
	if (GetType() != SMS_Submit) return FALSE;

	if (x==TRUE) {
		firstbyte |= 0x20;
	} else {
		firstbyte &= 0xDF;//clear bit
	}
	return TRUE;
}

GSM_Error GSM_SMSEntry::SetDateTime(GSM_DateTime *DT)
{
	return SetSMSDateTime(DT,DateTime);
}

GSM_Error GSM_SMSEntry::GetSMSCTime(GSM_DateTime *DT)
{
	return GetSMSDateTime(DT,SMSCTime);
}

GSM_Error GSM_SMSEntry::SetSMSCTime(GSM_DateTime *DT)
{
	return SetSMSDateTime(DT,SMSCTime);
}

/* GSM 03.40 section 9.2.3.25 (TP Reject Duplicates) */
BOOLEAN GSM_SMSEntry::SetRejectDuplicates(BOOLEAN enable)
{
	if (GetType() != SMS_Submit) return FALSE;

	firstbyte &= 0xFB;
	if (enable==TRUE) firstbyte+=0x04;
	return TRUE;
}
				
BOOLEAN GSM_SMSEntry::SetReplace(int i)
{
	if (GetType() == SMS_Report) return FALSE;

	if (i == -1) {
		TPPID=0;
	} else {
		if (i<-1 || i>7) return FALSE;
		TPPID=0x40+i;
	}
	return TRUE;
}

BOOLEAN GSM_SMSEntry::GetReplace(int *i)
{
	if (GetType() == SMS_Report) return FALSE;

	(*i) = -1;
	if (TPPID >= 0x40 && TPPID < 0x48) (*i) = TPPID - 0x40;
	return TRUE;
}

GSM_Error GSM_SMSEntry::GetUDH(unsignedstring *UDH)
{
	unsigned int Len;

	UDH->clear();
	if (firstbyte & 0x40) {
		if (UserData.size()==0) return GSM_Return_Error(GSM_ERR_UNKNOWN);
		Len = UserData.data()[0];
		if (Len>UserData.size()) return GSM_Return_Error(GSM_ERR_UNKNOWN);
		UDH->append((const unsigned char *)UserData.data(),Len+1);
//		printf("UDH header %i\n",Len);
//		for (i=0;i<UDH->length();i++) printf("%02x ",UDH->data()[i]);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::SetUDH(unsignedstring *Text) 
{
	unsignedstring TMP;
int p,w;

	GetUDH(&TMP);
	if (TMP.size() != 0) {
		//delete old UDH
	} else {
		TMP.append((const unsigned char *)UserData.data(),UserData.size());
	}

	firstbyte |= 0x40;

	UserData.clear();
	UserData.append((const unsigned char *)Text->data(),Text->size());
	UserData.append((const unsigned char *)TMP.data(),TMP.size());

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		TPUDL+=Text->size();
		break;
	case SMS_Coding_Default_No_Compression:
		p = 0;
		do {
			p+=7;
			w=(p-Text->size())%p;
		} while (w<0);
		TPUDL+=(Text->size()*8+w)/7;
//		TPUDL+=(Text->size());
		break;
	case SMS_Coding_8bit:
		TPUDL+=Text->size();
		break;
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::GetSMSCNumber(wchar_t *Destination) 
{
	return GSM_DecodeSMSNumber(Destination,SMSCNumber.data(),SMSCNumber.size(),FALSE);
}

void GSM_SMSEntry::SetSMSCNumber(wchar_t *Text) 
{
	SMSCNumber.clear();
	GSM_EncodeSMSNumber(&SMSCNumber, Text, FALSE);
}

/* See GSM 03.40 section 9.2.3.11 */
GSM_Error GSM_SMSEntry::GetSMSDateTime(GSM_DateTime *DT, const unsigned char *buffer)
{
	DT->Year 	= BCDCharToChar(buffer[0]);
	DT->Month 	= BCDCharToChar(buffer[1]);
	DT->Day 	= BCDCharToChar(buffer[2]);
	DT->Hour 	= BCDCharToChar(buffer[3]);
	DT->Minute 	= BCDCharToChar(buffer[4]);
	DT->Second 	= BCDCharToChar(buffer[5]);
	if (DT->Year < 90) DT->Year  += 2000;
	if (DT->Year < 100) DT->Year += 1900;

	return GSM_Return_Error(GSM_ERR_NONE);
}

/* See GSM 03.40 section 9.2.3.11 */
GSM_Error GSM_SMSEntry::SetSMSDateTime(GSM_DateTime *DT, unsigned char *buffer)
{
	buffer[0] = CharToBCD(DT->Year-2000);
	buffer[1] = CharToBCD(DT->Month);
	buffer[2] = CharToBCD(DT->Day);
	buffer[3] = CharToBCD(DT->Hour);
	buffer[4] = CharToBCD(DT->Minute);
	buffer[5] = CharToBCD(DT->Second);
	buffer[6] = 0;

	return GSM_Return_Error(GSM_ERR_NONE);
}

//----------------------------------------------------------------------------

GSM_SMSC::GSM_SMSC()
{
	Name 		= NULL;
	DefaultNumber 	= NULL;
	SMSCNumber 	= NULL;
}

GSM_SMSC::~GSM_SMSC()
{
	delete(Name);
	delete(DefaultNumber);
	delete(SMSCNumber);
}

void GSM_SMSC::SetName(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	Name = (wchar_t *)realloc(Name,len);
	memcpy(Name,N,len);
}

wchar_t *GSM_SMSC::GetName()
{
	if (Name == NULL) return 0x00;
	return Name;
}

void GSM_SMSC::SetSMSCNumber(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	SMSCNumber = (wchar_t *)realloc(SMSCNumber,len);
	memcpy(SMSCNumber,N,len);
}

wchar_t *GSM_SMSC::GetSMSCNumber()
{
	if (SMSCNumber == NULL) return 0x00;
	return SMSCNumber;
}

void GSM_SMSC::SetDefaultNumber(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	DefaultNumber = (wchar_t *)realloc(DefaultNumber,len);
	memcpy(DefaultNumber,N,len);
}

wchar_t *GSM_SMSC::GetDefaultNumber()
{
	if (DefaultNumber == NULL) return 0x00;
	return DefaultNumber;
}

//-----------------------------------------------------------------------------

GSM_Error GSM_SMSNumbersSubEntry::GetPhoneNumber(wchar_t *Destination)
{
	return GSM_DecodeSMSNumber(Destination,PhoneNumber.data(),PhoneNumber.size(),TRUE);
}

void GSM_SMSNumbersSubEntry::SetPhoneNumber(wchar_t *Text) 
{
	PhoneNumber.clear();
	GSM_EncodeSMSNumber(&PhoneNumber, Text, TRUE);
}

GSM_SMSNumbersSubEntry::GSM_SMSNumbersSubEntry()
{
	Next = NULL;
}

GSM_SMSNumbersSubEntry::~GSM_SMSNumbersSubEntry()
{
	delete(Next);
}

void GSM_SMSNumbersSubEntry::SetNext(GSM_SMSNumbersSubEntry *Nxt)
{
	Next = Nxt;
}

GSM_SMSNumbersSubEntry *GSM_SMSNumbersSubEntry::GetNext()
{
	return Next;
}

/* ------------------------------------------------------------------------ */

GSM_SMSNumbers::GSM_SMSNumbers()
{
	Entries = NULL;
}

GSM_SMSNumbers::~GSM_SMSNumbers()
{
	ClearAll();
}

BOOLEAN GSM_SMSNumbers::GetNext(GSM_SMSNumbersSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

GSM_Error GSM_SMSNumbers::Add(wchar_t *Text)
{
	GSM_SMSNumbersSubEntry *Entry,*Entry2;

	Entry = new GSM_SMSNumbersSubEntry;
	Entry->SetPhoneNumber(Text);

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSNumbers::Add(const unsigned char *Buffer, const int Len)
{
	GSM_SMSNumbersSubEntry *Entry,*Entry2;

	Entry = new GSM_SMSNumbersSubEntry;
	Entry->PhoneNumber.append(Buffer,Len);

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

void GSM_SMSNumbers::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}

// ---------------------------- SMSListSubEntry -------------------------------

GSM_SMSListSubEntry::GSM_SMSListSubEntry()
{
	Next = NULL;
}

GSM_SMSListSubEntry::~GSM_SMSListSubEntry()
{
	delete(Next);
}

void GSM_SMSListSubEntry::SetSMS(GSM_SMSEntry *SMS2)
{
	GSM_SMSNumbersSubEntry *Number;

	SMS.PhoneNumbers.ClearAll();
	Number = NULL;
	while (SMS2->PhoneNumbers.GetNext(&Number)) {
		SMS.PhoneNumbers.Add(Number->PhoneNumber.data(),Number->PhoneNumber.size());
	}

	SMS.UserData.clear();
	SMS.UserData.append(SMS2->UserData.data(),SMS2->UserData.size());

	SMS.SMSCNumber.clear();
	SMS.SMSCNumber.append(SMS2->SMSCNumber.data(),SMS2->SMSCNumber.size());

	SMS.TPDCS 	  = SMS2->TPDCS;
	memcpy(SMS.DateTime,SMS2->DateTime,7);
	memcpy(SMS.SMSCTime,SMS2->SMSCTime,7);
	SMS.TPStatus  	  = SMS2->TPStatus;
	SMS.TPUDL	  = SMS2->TPUDL;
	SMS.TPVP 	  = SMS2->TPVP;
	SMS.firstbyte 	  = SMS2->firstbyte;
	SMS.TPMR 	  = SMS2->TPMR;
	SMS.TPPID 	  = SMS2->TPPID;

	SMS.ID 		= SMS2->ID;
	SMS.Memory 	  = SMS2->Memory;
	SMS.Icon 	  = SMS2->Icon;
	SMS.SaveDateTimeAvailable = SMS2->SaveDateTimeAvailable;
	if (SMS.SaveDateTimeAvailable) memcpy(&SMS.SaveDateTime,&SMS2->SaveDateTime,sizeof(GSM_DateTime));
	SMS.Name.clear();
	SMS.Name.append(SMS2->Name.data(),SMS2->Name.length());
}

GSM_SMSEntry *GSM_SMSListSubEntry::GetSMS()
{
	return &SMS;
}

void GSM_SMSListSubEntry::SetNext(GSM_SMSListSubEntry *Nxt)
{
	Next = Nxt;
}

GSM_SMSListSubEntry *GSM_SMSListSubEntry::GetNext()
{
	return Next;
}

/* ------------------------------------------------------------------------ */

GSM_SMSList::GSM_SMSList()
{
	Entries = NULL;
}

GSM_SMSList::~GSM_SMSList()
{
	delete(Entries);
}

BOOLEAN GSM_SMSList::GetNext(GSM_SMSListSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

GSM_Error GSM_SMSList::Add(GSM_SMSEntry *SMS)
{
	GSM_SMSListSubEntry *Entry,*Entry2;

	Entry = new GSM_SMSListSubEntry;
	Entry->SetSMS(SMS);

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

void GSM_SMSList::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}
