/* Martin Strigl */

#include <sys/file.h>
#include <sys/time.h>
#include <string.h>
#include <termios.h>
#include <errno.h>

#include "ser_unx.h"

#ifndef O_NONBLOCK
#  define O_NONBLOCK  0
#endif

#ifdef __NetBSD__
#  define FNONBLOCK O_NONBLOCK

#  define  B57600   0010001
#  define  B115200  0010002
#  define  B230400  0010003
#  define  B460800  0010004
#  define  B500000  0010005
#  define  B576000  0010006
#  define  B921600  0010007
#  define  B1000000 0010010
#  define  B1152000 0010011
#  define  B1500000 0010012
#  define  B2000000 0010013
#  define  B2500000 0010014
#  define  B3000000 0010015
#  define  B3500000 0010016
#  define  B4000000 0010017
#endif

typedef struct {
	speed_t code;
	int	value;
} baud_record;

#ifdef B19200
#  define SERIAL_DEFAULT_SPEED 19200
#else
#  define SERIAL_DEFAULT_SPEED 9600
#endif

static baud_record baud_table[] = {
	{ B50,		50 },
	{ B75,		75 },
	{ B110,		110 },
	{ B134,		134 },
	{ B150,		150 },
	{ B200,		200 },
	{ B300,		300 },
	{ B600,		600 },
	{ B1200,	1200 },
	{ B1800,	1800 },
	{ B2400,	2400 },
	{ B4800,	4800 },
	{ B9600,	9600 },
#ifdef B19200
	{ B19200,	19200 },
#else /* ! defined (B19200) */
#ifdef EXTA
	{ EXTA,		19200 },
#endif /* EXTA */
#endif /* ! defined (B19200) */
#ifdef B38400
	{ B38400,	38400 },
#else /* ! defined (B38400) */
#ifdef EXTB
	{ EXTB,		38400 },
#endif /* EXTB */
#endif /* ! defined (B38400) */
#ifdef B57600
	{ B57600,	57600 },
#endif
#ifdef B76800
	{ B76800,	76800 },
#endif
#ifdef B115200
	{ B115200,	115200 },
#endif
#ifdef B230400
	{ B230400,	230400 },
#else
#ifdef _B230400
	{ _B230400,	230400 },
#endif /* _B230400 */
#endif /* ! defined (B230400) */
#ifdef B460800
	{ B460800,	460800 },
#else
#ifdef _B460800
	{ _B460800,	460800 },
#endif /* _B460800 */
#endif /* ! defined (B460800) */
#ifdef B500000
	{ B500000,	500000 },
#endif
#ifdef B576000
	{ B576000,	576000 },
#endif
#ifdef B921600
	{ B921600,	921600 },
#endif
#ifdef B1000000
	{ B1000000,	1000000 },
#endif
#ifdef B1152000
	{ B1152000,	1152000 },
#endif
#ifdef B1500000
	{ B1500000,	1500000 },
#endif
#ifdef B2000000
	{ B2000000,	2000000 },
#endif
#ifdef B2500000
	{ B2500000,	2500000 },
#endif
#ifdef B3000000
	{ B3000000,	3000000 },
#endif
#ifdef B3500000
	{ B3500000,	3500000 },
#endif
#ifdef B4000000
	{ B4000000,	4000000 },
#endif
	{ B0,		0 },
};

typedef struct {
	tcflag_t	code;
	int		value;
} bitmask_record;

static bitmask_record bitmask_table[] = {
	{ CS5,		5 },
	{ CS6,		6 },
	{ CS7,		7 },
	{ CS8,		8 },
	{ CS8,		0 },
};

#define SERIAL_DEFAULT_BITMASK 8


GSM_Error GSM_Device_Serial::Open(char *Dev, char *Prot, char *Pho, char **DeviceModel, GSM_AllPhones *Phones)
{
	int			i;
	struct termios		t;
	
	/* O_NONBLOCK MUST is required to avoid waiting for DCD */
	Device = open(Dev, O_RDWR | O_NOCTTY | O_NONBLOCK);
	if (Device < 0) {
		(*Debug)->Error("SerialOpen");
		return GSM_Return_Error(GSM_ERR_DEVICE_OPEN);
	}

#ifdef TIOCEXCL
	/* open() calls from other applications shall fail now */
	ioctl(Device, TIOCEXCL, (char *) 0);
#endif

	if (tcgetattr(Device, &old_settings) == -1) {
		close(Device);
		(*Debug)->Error("SerialOpen: tcgetattr");
		return GSM_Return_Error(GSM_ERR_DEVICE_READ);
    	}

    	if (tcflush(Device, TCIOFLUSH) == -1) {
		Close();
		(*Debug)->Error("SerialOpen: tcflush");
		return GSM_Return_Error(GSM_ERR_DEVICE_OPEN);
    	}
	
	memcpy(&t, &old_settings, sizeof(struct termios));

	/* Opening without parity */
    	t.c_iflag       = IGNPAR;
    	t.c_oflag       = 0;
    	/* disconnect line, 8 bits, enable receiver,
     	 * ignore modem lines,lower modem line after disconnect
     	 */
    	t.c_cflag       = B0 | CS8 | CREAD | CLOCAL | HUPCL;
    	/* enable hardware (RTS/CTS) flow control (NON POSIX) 	*/
    	/* t.c_cflag 	|= CRTSCTS;			  	*/
    	t.c_lflag       = 0;
    	t.c_cc[VMIN]    = 1;
    	t.c_cc[VTIME]   = 0;

    	if (tcsetattr(Device, TCSANOW, &t) == -1) {
		Close();
		(*Debug)->Error("SerialOpen: tcsetattr");
		return GSM_Return_Error(GSM_ERR_DEVICE_OPEN);
    	}

    	/* Making file descriptor asynchronous. */
    	if (fcntl(Device, F_SETFL, FASYNC | FNONBLOCK) == -1) {
		Close();
		(*Debug)->Error("SerialOpen: fcntl");
		return GSM_Return_Error(GSM_ERR_DEVICE_OPEN);
    	}

	(*Debug)->Deb("[STATE     : GSM_Device_serial::Open -> device %s opened succesfully\n",Dev);
        return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Device_Serial::Close()
{
	tcsetattr(Device, TCSANOW, &old_settings);
	close(Device);
	(*Debug)->Deb("[STATE     : GSM_Device_serial::Close -> device closed succesfully\n");
        return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Device_Serial::Read(unsigned char *buf, int *len)
{
    	struct timeval  		timeout2;
    	fd_set	  			readfds;
    	int	     			actual = 0;

	(*Debug)->Deb("[STATE     : GSM_Device_serial::Read -> Starting to read from device\n");

    	FD_ZERO(&readfds);
    	FD_SET(Device, &readfds);

    	timeout2.tv_sec     = 0;
//    	timeout2.tv_usec    = 1;
    	timeout2.tv_usec    = 50000;

    	if (select(Device+1, &readfds, NULL, NULL, &timeout2)) {
		actual = read(Device, buf, *len);
		if (actual == -1) {
			(*Debug)->Error("SerialRead: read");
			return GSM_Return_Error(GSM_ERR_DEVICE_READ);
		}
    	}

	(*Debug)->Deb("[STATE     : GSM_Device_serial::Read -> Ending to read from device\n");

        return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Device_Serial::Write(const unsigned char *buf, int len)
{
    	int		     	ret;
    	size_t		  	actual = 0;

	(*Debug)->Deb("[STATE     : GSM_Device_serial::Write -> Starting to write to device\n");

    	do {
		ret = write(Device, (unsigned char *)buf, len - actual);
		if (ret < 0 && errno == EAGAIN) {
			my_sleep(1);
			continue;
                }
		if (ret < 0) {
	    		if (actual != len) {
				(*Debug)->Error("SerialWrite: write");
				return GSM_Return_Error(GSM_ERR_DEVICE_WRITE);
			}
		}
		actual  += ret;
		buf     += ret;
    		my_sleep(1);
    	} while (actual < len);

	(*Debug)->Deb("[STATE     : GSM_Device_serial::Write -> Ending to write to device\n");

        return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Device_Serial::SetParameters(int speed, int bits, bool parity, int stopbits)
{
	int	err_speed,err_bits,err_par,err_stop;

	err_speed=SetSpeed(speed);
	if (err_speed!=0) {
		(*Debug)->Error("SetParameters: SetSpeed");
	        return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}	
	err_bits=SetBits(bits);
	if (err_bits!=0) {
		(*Debug)->Error("SetParameters: SetBits");
	        return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}	
	err_par=SetParity(parity);
	if (err_par!=0) {
		(*Debug)->Error("SetParameters: SetParity");
	        return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}	
	err_stop=SetStopBits(stopbits);
	if (err_stop!=0) {
		(*Debug)->Error("SetParameters: SetStopBits");
	        return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}	

        return GSM_Return_Error(GSM_ERR_NONE);
}

int GSM_Device_Serial::SetSpeed(int speed)
{
    	struct termios  	t;
	baud_record		*curr = baud_table;


    	if (tcgetattr(Device, &t)) {
		(*Debug)->Error("SetSpeed: tcgetattr");
		return 1;
    	}

	while (curr->value != speed) {
		curr++;
		/* This is how we make default fallback */
		if (curr->value == 0) {
			if (speed == SERIAL_DEFAULT_SPEED) {
				(*Debug)->Error("SetSpeed: speed not supported");
				return 1;
			}
			curr = baud_table;
			speed = SERIAL_DEFAULT_SPEED;
		}
	}

	(*Debug)->Deb("[STATE     : GSM_Device_serial::SetSpeed -> setting speed to %d\n",curr->value);

    	cfsetispeed(&t, curr->code);
    	cfsetospeed(&t, curr->code);

    	if (tcsetattr(Device, TCSADRAIN, &t) == -1) {
		Close();
		(*Debug)->Error("SetSpeed: tcsetattr");
		return 1;
    	}

        return 0;
}

int GSM_Device_Serial::SetParity(bool parity)
{
    	struct termios	  	t;

    	if (tcgetattr(Device, &t)) {
		(*Debug)->Error("SetParity: tcgetattr");
		return 1;
    	}

    	if (parity) {
		t.c_cflag |= (PARENB | PARODD);
		t.c_iflag = 0;
		(*Debug)->Deb("[STATE     : GSM_Device_serial::SetParity -> enabling parity\n");
    	} else {
		(*Debug)->Deb("[STATE     : GSM_Device_serial::SetParity -> disabling parity\n");
		t.c_iflag = IGNPAR;
    	}

    	if (tcsetattr(Device, TCSANOW, &t) == -1){
		Close();
		(*Debug)->Error("SetParity: tcsetattr");
		return 1;
    	}

    	return 0;
}

int GSM_Device_Serial::SetStopBits(int stopbits)
{
    	struct termios	  	t;

    	if (tcgetattr(Device, &t)) {
		(*Debug)->Error("SetStopBits: tcgetattr");
		return 1;
    	}

    	if (stopbits == 2) {
		(*Debug)->Deb("[STATE     : GSM_Device_serial::SetStopBits -> setting 2 stopbits\n");
		t.c_cflag |= CSTOPB;
	} else {
		(*Debug)->Deb("[STATE     : GSM_Device_serial::SetStopBits -> leaving 1 stopbits\n");
    	}

    	if (tcsetattr(Device, TCSANOW, &t) == -1){
		Close();
		(*Debug)->Error("SetStopBits: tcsetattr");
		return 1;
    	}

    	return 0;
}

int GSM_Device_Serial::SetBits(int bits)
{
    	struct termios  	t;
	bitmask_record		*curr = bitmask_table;


    	if (tcgetattr(Device, &t)) {
		(*Debug)->Error("SetBits: tcgetattr");
		return 1;
    	}

	while (curr->value != bits) {
		curr++;
		/* This is how we make default fallback */
		if (curr->value == 0) {
			if (bits == SERIAL_DEFAULT_BITMASK) {
				(*Debug)->Error("SetBits: bitmask not supported");
				return 1;
			}
			curr = bitmask_table;
			bits = SERIAL_DEFAULT_BITMASK;
		}
	}

	(*Debug)->Deb("[STATE     : GSM_Device_serial::SetBits -> setting bitmask to %d\n",curr->value);

	t.c_cflag |= curr->code;

    	if (tcsetattr(Device, TCSANOW, &t) == -1) {
		Close();
		(*Debug)->Error("SetBits: tcsetattr");
		return 1;
    	}

        return 0;
}

GSM_Error GSM_Device_Serial::SetLines(bool dtr, bool rts)
{
    	struct termios	  	t;
    	unsigned int	    	flags;

    	if (tcgetattr(Device, &t)) {
		(*Debug)->Error("SerialSetLines: tcgetattr");
		return GSM_Return_Error(GSM_ERR_DEVICE_READ);
    	}

#ifdef CRTSCTS
    	/* Disabling hardware flow control */
    	t.c_cflag &= ~CRTSCTS;
#endif

    	if (tcsetattr(Device, TCSANOW, &t) == -1) {
		Close();
		(*Debug)->Error("SerialSetLines: tcsetattr");
		return GSM_Return_Error(GSM_ERR_DEVICE_OPEN);
    	}

    	flags = TIOCM_DTR;
    	if (dtr) {
		ioctl(Device, TIOCMBIS, &flags);
	} else {
		ioctl(Device, TIOCMBIC, &flags);
	}

    	flags = TIOCM_RTS;
    	if (rts) {
		ioctl(Device, TIOCMBIS, &flags);
	} else {
		ioctl(Device, TIOCMBIC, &flags);
	}

    	flags = 0;
    	ioctl(Device, TIOCMGET, &flags);

    	if (((flags&TIOCM_DTR)==TIOCM_DTR) != dtr) return GSM_Return_Error(GSM_ERR_DEVICE_WRITE);
    	if (((flags&TIOCM_RTS)==TIOCM_RTS) != rts) return GSM_Return_Error(GSM_ERR_DEVICE_WRITE);

        return GSM_Return_Error(GSM_ERR_NONE);
}
