/* (C) 2005 by Marcin Wiacek www.mwiacek.com */

#include "gsmsms.h"
#include "../gsmmisc.h"
#include "../../misc/coding/coding.h"

static GSM_SMS_UDH_Types UDHHeaders[] = {
{SMS_UDH_Linked_Short,6,"\x05\x00\x03\x01\x00\x00",3,-1,5,4},
{SMS_UDH_Linked_Long, 7,"\x06\x08\x04\x00\x00\x00\x00",-1,3,6,5},
{SMS_UDH_Linked_Long, 0,"",-1,-1,-1,-1}
};

GSM_Error GSM_SMSEntry::DecodeUDH(UDHList *List)
{
    int 			i=0,j,pos=0;
	unsignedstring 		UDH;
	GSM_Error 		error;
	BOOLEAN			wrong;
	GSM_SMS_UDH_Types	New;

	List->clear();

	error = GetUDH(&UDH);
	if (error.Code != GSM_ERR_NONE) return error;

	while (pos!=UDH.length()) {
		while(UDHHeaders[i].Length!=0) {
			wrong = FALSE;
			if ((int)(UDH.length()-pos)>=UDHHeaders[i].Length) {
				for (j=0;j<UDHHeaders[i].Length;j++) {
					if (UDHHeaders[i].ID8bit!=-1 && j==UDHHeaders[i].ID8bit) continue;
					if (UDHHeaders[i].PartNumber8bit!=-1 && j==UDHHeaders[i].PartNumber8bit) continue;
					if (UDHHeaders[i].AllParts8bit!=-1 && j==UDHHeaders[i].AllParts8bit) continue;
					if (UDHHeaders[i].ID16bit!=-1 && (j==UDHHeaders[i].ID16bit || j==UDHHeaders[i].ID16bit+1)) continue;
					if (UDHHeaders[i].Text[j]==UDH.data()[j+pos]) continue;
					wrong = TRUE;
					break;
				}
			} else {
				wrong = TRUE;
			}
			if (!wrong) {
				memcpy(&New,&UDHHeaders[i],sizeof(GSM_SMS_UDH_Types));
				memcpy(New.Text,UDH.data()+pos,UDHHeaders[i].Length);
				if (New.ID8bit!=-1) New.ID8bit=New.Text[New.ID8bit];
				if (New.ID16bit!=-1) New.ID16bit=New.Text[New.ID16bit]*256+New.Text[New.ID16bit+1];
				if (New.AllParts8bit!=-1) New.AllParts8bit=New.Text[New.AllParts8bit];
				if (New.PartNumber8bit!=-1) New.PartNumber8bit=New.Text[New.PartNumber8bit];
				List->push_back(New);
				pos+=UDHHeaders[i].Length;
				break;
			}
			i++;
		}
		if (!wrong) continue;
		return GSM_Return_Error(GSM_ERR_NOT_SUPPORTED);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_SMSEntry::GSM_SMSEntry() 
{
	TPDCS 	  = 0;
	memset(DateTime,0,sizeof(DateTime));
	memset(SMSCTime,0,sizeof(SMSCTime));
	TPStatus  = 0;
	TPUDL	  = 0;
	TPVP 	  = 0;
	firstbyte = 0;
	TPMR 	  = 0;
	TPPID 	  = 0;

	Icon = SMS_UnRead;
	SaveDateTimeAvailable = FALSE;
}

GSM_SMSEntry::~GSM_SMSEntry() 
{
}

SMS_Type GSM_SMSEntry::GetType()
{
	switch (firstbyte & 0x03) {
		case  1: return SMS_Submit;
		case  2: return SMS_Report;
		default: return SMS_Deliver;
	}
	return SMS_Deliver;
}

//to test
SMS_Report_Status GSM_SMSEntry::GetReportStatus()
{
	/* GSM 03.40 section 9.2.3.15 (TP-ST) */	
	if ((TPStatus & 0x80) == 0x80) return SMS_Status_Unknown;

	if ((TPStatus & 0x7F) <= 2) return SMS_Status_Delivered;
	if ((TPStatus & 0x7F) >= 0x20 &&
	    (TPStatus & 0x7F) <= 0x3F) return SMS_Status_Pending;
	if ((TPStatus & 0x7F) >= 0x40) return SMS_Status_Failed;
	return SMS_Status_Unknown;
}

GSM_Error GSM_SMSEntry::SetType(SMS_Type Type)
{
	/* bits 0 and 1 */
	firstbyte &= 0xFC;
	switch (Type) {
		case SMS_Deliver: 		  break;
		case SMS_Submit : firstbyte += 1; break;
		case SMS_Report : firstbyte += 2; break;
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

SMS_Coding_Type GSM_SMSEntry::GetCoding()
{
	if (GetType() == SMS_Report) return SMS_Coding_8bit;

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xC0) == 0) {
		/* bits 7..4 set to 00xx */
		if ((TPDCS & 0xC) == 0xC) {
			//reserved alphabet value in TPDCS
		} else {
			if (TPDCS == 0) 	    return SMS_Coding_Default_No_Compression;
			if ((TPDCS & 0x2C) == 0x00) return SMS_Coding_Default_No_Compression;
			if ((TPDCS & 0x2C) == 0x20) return SMS_Coding_Default_Compression;
			if ((TPDCS & 0x2C) == 0x08) return SMS_Coding_Unicode_No_Compression;
			if ((TPDCS & 0x2C) == 0x28) return SMS_Coding_Unicode_Compression;
		}
	} else if ((TPDCS & 0xF0) >= 0x40 && (TPDCS & 0xF0) <= 0xB0) {
		/* bits 7..4 set to 0100 ... 1011 */
		//reserved coding group in TPDCS
	} else if (((TPDCS & 0xF0) == 0xC0) || ((TPDCS & 0xF0) == 0xD0)) {
		/* bits 7..4 set to 1100 or 1101 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
		} else {
			return SMS_Coding_Default_No_Compression;
		}
	} else if ((TPDCS & 0xF0) == 0xE0) {
		/* bits 7..4 set to 1110 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
		} else {
			return SMS_Coding_Unicode_No_Compression;
		}
	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
		} else {
			if ((TPDCS & 4) == 0) return SMS_Coding_Default_No_Compression;
		}
	}	
	return SMS_Coding_8bit;
}

//to test
GSM_Error GSM_SMSEntry::SetCoding(SMS_Coding_Type Coding)
{
	if (GetType() == SMS_Report) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xC0) == 0) {
		/* bits 7..4 set to 00xx */
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			TPDCS &= 0xD3; 	// set bits 2 and 5 to 0
			TPDCS += 8; 	// set bit 3 to 1
			break;
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			TPDCS &= 0xD3; 	// set bits 2, 3 and 5 to 0
			break;
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			TPDCS &= 0xF3; 	// set bit 3 to 0
			TPDCS += 4; 	// set bit 3 to 1
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		}
	} else if ((TPDCS & 0xF0) >= 0x40 && (TPDCS & 0xF0) <= 0xB0) {
		/* bits 7..4 set to 0100 ... 1011 */
		//reserved coding group in TPDCS
		return GSM_Return_Error(GSM_ERR_UNKNOWN);
	} else if (((TPDCS & 0xF0) == 0xC0) || ((TPDCS & 0xF0) == 0xD0)) {
		/* bits 7..4 set to 1100 or 1101 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		}
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			break;
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);			
		}
	} else if ((TPDCS & 0xF0) == 0xE0) {
		/* bits 7..4 set to 1110 */
		if ((TPDCS & 4) == 4) {
			//set reserved bit 2 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		}
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			break;
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);			
		}
    	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);	
		}
		switch (Coding) {
		case SMS_Coding_Unicode_No_Compression:
			return GSM_Return_Error(GSM_ERR_UNKNOWN);				
		case SMS_Coding_Unicode_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_Default_No_Compression:
			TPDCS &= 0xEF; 	// set bit 4 to 0
			break;
		case SMS_Coding_Default_Compression:
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		case SMS_Coding_8bit:
			TPDCS &= 0xEF; 	// set bit 4 to 0
			TPDCS += 0x10;  // set bit 4 to 1
		}
	}	
	return GSM_Return_Error(GSM_ERR_NONE);
}

int GSM_SMSEntry::GetClass() 
{
	if (GetType() == SMS_Report) return -1;

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xD0) == 0x10) {
		/* bits 7..4 set to 00x1 */
		if ((TPDCS & 0xC) == 0xC) {
			//reserved alphabet value in TPDCS
		} else {
			return (TPDCS & 3);
		}
	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
		} else {
			return (TPDCS & 3);
		}
	}
	return -1;
}

//to test
GSM_Error GSM_SMSEntry::SetClass(int Class) 
{
	if (GetType() == SMS_Report) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	if (Class < -1 || Class > 3) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);

	/* GSM 03.40 section 9.2.3.10 (TP-Data-Coding-Scheme) and GSM 03.38 section 4 */
	if ((TPDCS & 0xC0) == 0) {
		/* bits 7..4 set to 00xx */
		if ((TPDCS & 0xC) == 0xC) {
			//reserved alphabet value in TPDCS
			return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		}
		TPDCS &= 0xEF; //set bit 4 to 0
		if (Class == -1) return GSM_Return_Error(GSM_ERR_NONE);
		TPDCS += 0x10; //set bit 4 to 1
    	} else if ((TPDCS & 0xF0) == 0xF0) {
		/* bits 7..4 set to 1111 */
		if ((TPDCS & 8) == 8) {
			//set reserved bit 3 in TPDCS
			return GSM_Return_Error(GSM_ERR_UNKNOWN);	
		}
		if (Class == -1) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	} else {
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}
	TPDCS += Class;

	return GSM_Return_Error(GSM_ERR_NONE);
}

//todo
int GSM_SMSEntry::GetRelativeValidity() 
{
	return -1;
}

//todo
GSM_Error GSM_SMSEntry::SetRelativeValidity()
{
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::GetDecodedText(wchar_t *text)
{
	unsignedstring  	TMP, UDH;
	unsigned int		i = 0;
	unsigned char 		*Buffer;
	GSM_Error		error;

	error = GetText(&TMP);
	if (error.Code != GSM_ERR_NONE) return error;

	Buffer=(unsigned char *)TMP.data();

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		if ((TMP.size() % 2) != 0) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		for (i=0;i<TMP.size()/2;i++) {
			text[i] = Buffer[i*2]*256 + Buffer[i*2+1];
		}
		text[i] = 0;
		return GSM_Return_Error(GSM_ERR_NONE);
	case SMS_Coding_Default_No_Compression:
		error = GetUDH(&UDH);
		GSM_DecodeSMSText(text, Buffer, TPUDL, TMP.size(), UDH.size());
		return GSM_Return_Error(GSM_ERR_NONE);
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
	default:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}
}

GSM_Error GSM_SMSEntry::SetDecodedText(wchar_t *Text)
{
	unsignedstring  TMP,TMP2;
	unsigned char	Buffer[200];
	int		i,j;
	GSM_Error	error;

	error = GetUDH(&TMP2);
	if (error.Code != GSM_ERR_NONE) return error;

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		//fixme
		for (i=0;i<UnicodeLength(Text);i++) {
			TMP.push_back(Text[i]/256);
			TMP.push_back(Text[i]%256);
		}
		return SetText(&TMP);
	case SMS_Coding_Default_No_Compression:
//		error = GSM_EncodeSMSText(Buffer, Text, &j, 133-TMP2.size(), TMP2.size());
		error = GSM_EncodeSMSText(Buffer, Text, &j, 140-TMP2.size(), TMP2.size());
		if (error.Code != GSM_ERR_NONE) return error;
		for (i=0;i<(j*7/8);i++) TMP.push_back(Buffer[i]);
printf("%i\n",(j*7/8));
		return SetText(&TMP);		
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	default:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}
}

GSM_Error GSM_SMSEntry::GetText(unsignedstring *Text) 
{
	unsignedstring 	TMP;
	GSM_Error	error;

	error = GetUDH(&TMP);
	if (error.Code != GSM_ERR_NONE) return error;

	Text->append((const unsigned char *)UserData.data()+TMP.size(),UserData.size()-TMP.size());

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::SetText(unsignedstring *Text)
{
	unsignedstring 	TMP;
	int		Len;
	GSM_Error	error;

	error = GetUDH(&TMP);
	if (error.Code != GSM_ERR_NONE) return error;

	Len = TMP.size() + Text->size();
	if (Len > 140) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);

	switch (GetCoding()) {
	case SMS_Coding_Unicode_No_Compression:
		if ((Text->size() % 2) != 0) return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
		TPUDL = Len;
		break;
	case SMS_Coding_Default_No_Compression:
		TPUDL = Len;
		break;
	case SMS_Coding_Unicode_Compression:
	case SMS_Coding_Default_Compression:
	case SMS_Coding_8bit:
		return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
	}

	UserData.clear();
	UserData.append((const unsigned char *)TMP.data(),TMP.size());
	UserData.append((const unsigned char *)Text->data(),Text->size());

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::GetDateTime(GSM_DateTime *DT)
{
	return GetSMSDateTime(DT,DateTime);
}

BOOLEAN	GSM_SMSEntry::SenderSMSCReply()
{
	if (GetType() == SMS_Report) return FALSE;

	if ((firstbyte & 0x80)!=0) return TRUE;
	return FALSE;
}

GSM_Error GSM_SMSEntry::SetDateTime(GSM_DateTime *DT)
{
	return SetSMSDateTime(DT,DateTime);
}

GSM_Error GSM_SMSEntry::GetSMSCTime(GSM_DateTime *DT)
{
	return GetSMSDateTime(DT,SMSCTime);
}

GSM_Error GSM_SMSEntry::SetSMSCTime(GSM_DateTime *DT)
{
	return SetSMSDateTime(DT,SMSCTime);
}
				
//tocheck
GSM_Error GSM_SMSEntry::GetUDH(unsignedstring *UDH)
{
	unsigned int Len;

	UDH->clear();
	if (firstbyte & 0x40) {
		if (UserData.size()==0) return GSM_Return_Error(GSM_ERR_UNKNOWN);
		Len = UserData.data()[0];
		if (Len>UserData.size()) return GSM_Return_Error(GSM_ERR_UNKNOWN);
		UDH->append((const unsigned char *)UserData.data(),Len+1);
//		printf("UDH header %i\n",Len);
//		for (i=0;i<UDH->length();i++) printf("%02x ",UDH->data()[i]);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

//tocheck
GSM_Error GSM_SMSEntry::SetUDH(unsignedstring *Text) 
{
	unsignedstring TMP;

	GetUDH(&TMP);
	//check new TPUDL
	if (TMP.size() != 0) {
		//delete old UDH
		//set TPUDL
	}
	TMP.append((const unsigned char *)UserData.data(),UserData.size());

	UserData.clear();

	UserData.append((const unsigned char *)Text->data(),Text->size());
	UserData.append((const unsigned char *)TMP.data(),TMP.size());
	//set TPUDL

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSEntry::GetSMSCNumber(wchar_t *Destination) 
{
	return GSM_DecodeSMSNumber(Destination,SMSCNumber.data(),SMSCNumber.size(),FALSE);
}

void GSM_SMSEntry::SetSMSCNumber(wchar_t *Text) 
{
	SMSCNumber.clear();
	GSM_EncodeSMSNumber(&SMSCNumber, Text, FALSE);
}

/* See GSM 03.40 section 9.2.3.11 */
GSM_Error GSM_SMSEntry::GetSMSDateTime(GSM_DateTime *DT, const unsigned char *buffer)
{
	DT->Year 	= BCDCharToChar(buffer[0]);
	DT->Month 	= BCDCharToChar(buffer[1]);
	DT->Day 	= BCDCharToChar(buffer[2]);
	DT->Hour 	= BCDCharToChar(buffer[3]);
	DT->Minute 	= BCDCharToChar(buffer[4]);
	DT->Second 	= BCDCharToChar(buffer[5]);
	if (DT->Year < 90) DT->Year  += 2000;
	if (DT->Year < 100) DT->Year += 1900;

	return GSM_Return_Error(GSM_ERR_NONE);
}

/* See GSM 03.40 section 9.2.3.11 */
GSM_Error GSM_SMSEntry::SetSMSDateTime(GSM_DateTime *DT, unsigned char *buffer)
{
	buffer[0] = CharToBCD(DT->Year-2000);
	buffer[1] = CharToBCD(DT->Month);
	buffer[2] = CharToBCD(DT->Day);
	buffer[3] = CharToBCD(DT->Hour);
	buffer[4] = CharToBCD(DT->Minute);
	buffer[5] = CharToBCD(DT->Second);
	buffer[6] = 0;

	return GSM_Return_Error(GSM_ERR_NONE);
}

//-----------------------------------------------------------------------------

GSM_SMSListSubEntry::GSM_SMSListSubEntry()
{
	Next = NULL;
}

GSM_SMSListSubEntry::~GSM_SMSListSubEntry()
{
	delete(Next);
}

void GSM_SMSListSubEntry::SetSMS(GSM_SMSEntry *SMS2)
{
	GSM_SMSNumbersSubEntry *Number;

	SMS.PhoneNumbers.ClearAll();
	Number = NULL;
	while (SMS2->PhoneNumbers.GetNext(&Number)) {
		SMS.PhoneNumbers.Add(Number->PhoneNumber.data(),Number->PhoneNumber.size());
	}

	SMS.UserData.clear();
	SMS.UserData.append(SMS2->UserData.data(),SMS2->UserData.size());

	SMS.SMSCNumber.clear();
	SMS.SMSCNumber.append(SMS2->SMSCNumber.data(),SMS2->SMSCNumber.size());

	SMS.TPDCS 	  = SMS2->TPDCS;
	memcpy(SMS.DateTime,SMS2->DateTime,7);
	memcpy(SMS.SMSCTime,SMS2->SMSCTime,7);
	SMS.TPStatus  	  = SMS2->TPStatus;
	SMS.TPUDL	  = SMS2->TPUDL;
	SMS.TPVP 	  = SMS2->TPVP;
	SMS.firstbyte 	  = SMS2->firstbyte;
	SMS.TPMR 	  = SMS2->TPMR;
	SMS.TPPID 	  = SMS2->TPPID;

	SMS.ID 		= SMS2->ID;
	SMS.Memory 	  = SMS2->Memory;
	SMS.Icon 	  = SMS2->Icon;
	SMS.SaveDateTimeAvailable = SMS2->SaveDateTimeAvailable;
	if (SMS.SaveDateTimeAvailable) memcpy(&SMS.SaveDateTime,&SMS2->SaveDateTime,sizeof(GSM_DateTime));
	SMS.Name.clear();
	SMS.Name.append(SMS2->Name.data(),SMS2->Name.length());
}

GSM_SMSEntry *GSM_SMSListSubEntry::GetSMS()
{
	return &SMS;
}

void GSM_SMSListSubEntry::SetNext(GSM_SMSListSubEntry *Nxt)
{
	Next = Nxt;
}

GSM_SMSListSubEntry *GSM_SMSListSubEntry::GetNext()
{
	return Next;
}

/* ------------------------------------------------------------------------ */

GSM_SMSList::GSM_SMSList()
{
	Entries = NULL;
}

GSM_SMSList::~GSM_SMSList()
{
	delete(Entries);
}

BOOLEAN GSM_SMSList::GetNext(GSM_SMSListSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

GSM_Error GSM_SMSList::Add(GSM_SMSEntry *SMS)
{
	GSM_SMSListSubEntry *Entry,*Entry2;

	Entry = new GSM_SMSListSubEntry;
	Entry->SetSMS(SMS);

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

void GSM_SMSList::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}

//----------------------------------------------------------------------------

GSM_SMSC::GSM_SMSC()
{
	Name 		= NULL;
	DefaultNumber 	= NULL;
	SMSCNumber 	= NULL;
}

GSM_SMSC::~GSM_SMSC()
{
	delete(Name);
	delete(DefaultNumber);
	delete(SMSCNumber);
}

void GSM_SMSC::SetName(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	Name = (wchar_t *)realloc(Name,len);
	memcpy(Name,N,len);
}

wchar_t *GSM_SMSC::GetName()
{
	if (Name == NULL) return 0x00;
	return Name;
}

void GSM_SMSC::SetSMSCNumber(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	SMSCNumber = (wchar_t *)realloc(SMSCNumber,len);
	memcpy(SMSCNumber,N,len);
}

wchar_t *GSM_SMSC::GetSMSCNumber()
{
	if (SMSCNumber == NULL) return 0x00;
	return SMSCNumber;
}

void GSM_SMSC::SetDefaultNumber(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	DefaultNumber = (wchar_t *)realloc(DefaultNumber,len);
	memcpy(DefaultNumber,N,len);
}

wchar_t *GSM_SMSC::GetDefaultNumber()
{
	if (DefaultNumber == NULL) return 0x00;
	return DefaultNumber;
}

//-----------------------------------------------------------------------------

GSM_Error GSM_SMSNumbersSubEntry::GetPhoneNumber(wchar_t *Destination)
{
	return GSM_DecodeSMSNumber(Destination,PhoneNumber.data(),PhoneNumber.size(),TRUE);
}

void GSM_SMSNumbersSubEntry::SetPhoneNumber(wchar_t *Text) 
{
	PhoneNumber.clear();
	GSM_EncodeSMSNumber(&PhoneNumber, Text, TRUE);
}

GSM_SMSNumbersSubEntry::GSM_SMSNumbersSubEntry()
{
	Next = NULL;
}

GSM_SMSNumbersSubEntry::~GSM_SMSNumbersSubEntry()
{
	delete(Next);
}

void GSM_SMSNumbersSubEntry::SetNext(GSM_SMSNumbersSubEntry *Nxt)
{
	Next = Nxt;
}

GSM_SMSNumbersSubEntry *GSM_SMSNumbersSubEntry::GetNext()
{
	return Next;
}

/* ------------------------------------------------------------------------ */

GSM_SMSNumbers::GSM_SMSNumbers()
{
	Entries = NULL;
}

GSM_SMSNumbers::~GSM_SMSNumbers()
{
	ClearAll();
}

BOOLEAN GSM_SMSNumbers::GetNext(GSM_SMSNumbersSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

GSM_Error GSM_SMSNumbers::Add(wchar_t *Text)
{
	GSM_SMSNumbersSubEntry *Entry,*Entry2;

	Entry = new GSM_SMSNumbersSubEntry;
	Entry->SetPhoneNumber(Text);

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_SMSNumbers::Add(const unsigned char *Buffer, const int Len)
{
	GSM_SMSNumbersSubEntry *Entry,*Entry2;

	Entry = new GSM_SMSNumbersSubEntry;
	Entry->PhoneNumber.append(Buffer,Len);

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

void GSM_SMSNumbers::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}
