/* (C) 2003 - 2007 by Marcin Wiacek www.mwiacek.com */

#include <stdio.h>

#include "../cfg/config.h"
#include "gsmstate.h"
#include "misc/coding/coding.h"
#include "misc/files.h"

#ifdef WIN32
#  include "device/serial/ser_w32.h"
#  include "device/bluetoth/blue_w32.h"
#else
#  include "device/serial/ser_unx.h"
#endif
#include "device/irda/irda.h"

#include "protocol/nokia/fbus2.h"
#include "protocol/nokia/phonet.h"
#include "protocol/obex/obex.h"

#include "phone/nokia/dct3/n6110.h"
#include "phone/nokia/dct3/n7110.h"
#include "phone/nokia/dct4s40/n6510.h"
#include "phone/nokia/nauto.h"

GSM_StateMachine::GSM_StateMachine(FILE *deb_file, char *FileName, bool enable)
{
	Debug = new DebugInfo;

	Debug->SetDebug(deb_file,FileName,enable);

        Devices         = new GSM_AllDevices(&Debug);
        Protocols       = new GSM_AllProtocols(&Debug);
        Phones          = new GSM_AllPhones(&Debug);

	Debug->Deb("[GAMMU+    : %s built %s %s",VERSION,__TIME__,__DATE__);
	if (strlen(GetCompiler()) != 0) {
		Debug->Deb(" in %s]\n",GetCompiler());
	} else {
		Debug->Deb("]\n");
	}
	if (strlen(GetOS()) != 0) Debug->Deb("[RUN ON    : %s]\n",GetOS());

        Devices->Add(new GSM_Device_Infrared(&Debug));
        Devices->Add(new GSM_Device_Serial(&Debug));
#ifdef WIN32
        Devices->Add(new GSM_Device_Bluetooth(&Debug));
#endif

        Protocols->Add(new GSM_Protocol_FBUS2(&(Devices->Current),&Debug));
        Protocols->Add(new GSM_Protocol_PHONET(&(Devices->Current),&Debug));
        Protocols->Add(new GSM_Protocol_OBEX(&(Devices->Current),&Debug));
  
        Phones->Add(new GSM_Phone_N6110(&Debug,&(Devices->Current),&Protocols,&Phones));
        Phones->Add(new GSM_Phone_N7110(&Debug,&(Devices->Current),&Protocols,&Phones));
        Phones->Add(new GSM_Phone_N6510(&Debug,&(Devices->Current),&Protocols,&Phones));
        Phones->Add(new GSM_Phone_NAUTO(&Debug,&(Devices->Current),&Protocols,&Phones));
}

GSM_StateMachine::~GSM_StateMachine()
{
	Close();

	delete(Phones);
	delete(Protocols);
	delete(Devices);

	delete(Debug);
}

GSM_Error GSM_StateMachine::Close()
{
        GSM_Error error;

        if (Phones->Current != NULL) {
		Debug->Deb("[CLOSING PHONES]\n");
                error=Phones->Current->Close();
                if (error.Code != GSM_ERR_NONE) return error;
        }

        if (Protocols->Current != NULL) {
		Debug->Deb("[CLOSING PROTOCOLS]\n");
                error=Protocols->Current->Close();
                if (error.Code != GSM_ERR_NONE) return error;
        }

        if (Devices->Current != NULL) {
		Debug->Deb("[CLOSING DEVICES]\n");
	        error=Devices->Current->Close();
                if (error.Code != GSM_ERR_NONE) return error;
        }

        return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_StateMachine::Open(GSM_GPlusConfig *CFG)
{
        GSM_Error       			error;
        char            			*DeviceName = NULL;
	unsigned char   			Buff[100];
	list<GSM_Protocol_Info>::iterator 	proinfo;
	unsignedstring				Prot2,Pho2;
	int					i;

	for (i=0;i<(int)strlen(CFG->Protocol);i++) Prot2.push_back(tolower(CFG->Protocol[i]));
	for (i=0;i<(int)strlen(CFG->Phone);i++) Pho2.push_back(tolower(CFG->Phone[i]));

	Debug->Deb("[STARTUP   : device \"%s\", protocol \"%s\", phone \"%s\"]\n",CFG->Device,CFG->Protocol,CFG->Phone);

        error=Devices->Switch((char *)Prot2.data(),Protocols,Phones,CFG->Device);
        if (error.Code != GSM_ERR_NONE) return error;

        error=Protocols->Switch((char *)Prot2.data(),Phones);
        if (error.Code != GSM_ERR_NONE) return error;

        error=Devices->Current->Open(CFG->Device,(char *)Prot2.data(),(char *)Pho2.data(),&DeviceName,Phones);
        if (error.Code != GSM_ERR_NONE) return error;

        error=Protocols->Current->Open((char *)Prot2.data());
        if (error.Code != GSM_ERR_NONE) return error;

	for (proinfo=Protocols->Current->Info.begin(); proinfo!=Protocols->Current->Info.end(); ++proinfo) {
		if (!strcmp((char *)Prot2.data(),(*proinfo).Protocol)) break;
        }

        if (CFG->Phone[0] == 0 && (*proinfo).CanUseDeviceName) {
		error=Phones->SwitchToDeviceName(DeviceName,(char *)Prot2.data(),CFG->Compatibility);
	} else {
                error=Phones->Switch((char *)Pho2.data(),(char *)Prot2.data(),CFG->Compatibility);
	}
	if (error.Code != GSM_ERR_NONE) return error;

        error=Phones->Current->Open("");
        if (error.Code != GSM_ERR_NONE) return error;

	/* We will get now phone info */
        error=Phones->Current->GetCodeNameModel(Buff);
        if (error.Code != GSM_ERR_NONE) return error;

        return GSM_Return_Error(GSM_ERR_NONE);
}

BOOLEAN GSM_StateMachine::ReadCfgFile(INI_File *File)
{
#ifndef WIN32
	char		tmp[500];
#endif
	BOOLEAN		read = FALSE;

	if (read==FALSE && File->ReadFile("gammurc")==TRUE) read = TRUE;
#ifndef WIN32
	strcpy(tmp,getenv("HOME"));
	strcat(tmp,"/.gammurc");
	if (read==FALSE && File->ReadFile(tmp)==TRUE) read = TRUE;
	if (read==FALSE && File->ReadFile("/etc/gammurc")==TRUE) read = TRUE;
#endif
	return read;
}

void GSM_StateMachine::ReadCfg(GSM_GPlusConfig *CFG)
{
	INI_File	File;
	wchar_t		*value, x[200];
	BOOLEAN		read;

	read = ReadCfgFile(&File);

	StringToUnicode("gammu",x);

	strcpy(CFG->Device,"com2:");
#ifndef WIN32
	strcpy(CFG->Device,"/dev/ttyS1");
#endif
	value = File.GetValue(x,StringToUnicodeReturn("port"));
	if (value != NULL) sprintf(CFG->Device,"%s",UnicodeToStringReturn(value));

	strcpy(CFG->Protocol,"fbus");
	value = File.GetValue(x,StringToUnicodeReturn("connection"));
	if (value != NULL) sprintf(CFG->Protocol,"%s",UnicodeToStringReturn(value));

	CFG->Phone[0] = 0;
	value = File.GetValue(x,StringToUnicodeReturn("model"));
	if (value != NULL) sprintf(CFG->Phone,"%s",UnicodeToStringReturn(value));

	CFG->Compatibility[0] = 0;
	value = File.GetValue(x,StringToUnicodeReturn("compatibility"));
	if (value != NULL) sprintf(CFG->Compatibility,"%s",UnicodeToStringReturn(value));

	CFG->RSSLevel[0] = 0;
	value = File.GetValue(x,StringToUnicodeReturn("rsslevel"));
	if (value != NULL) sprintf(CFG->RSSLevel,"%s",UnicodeToStringReturn(value));

	strcpy(CFG->LogLevel,"nothing");
	value = File.GetValue(x,StringToUnicodeReturn("logformat"));
	if (value != NULL) sprintf(CFG->LogLevel,"%s",UnicodeToStringReturn(value));

	strcpy(CFG->LogFile,"");
	value = File.GetValue(x,StringToUnicodeReturn("logfile"));
	if (value != NULL) sprintf(CFG->LogFile,"%s",UnicodeToStringReturn(value));

	strcpy(CFG->SyncTime,"no");
	value = File.GetValue(x,StringToUnicodeReturn("synchronizetime"));
	if (value != NULL) sprintf(CFG->SyncTime,"%s",UnicodeToStringReturn(value));
}

/**
 * method for saving configuration file for HDD
 */
void GSM_StateMachine::SaveCfg(GSM_GPlusConfig *CFG)
{
	INI_File	File;
	wchar_t		x[200], x2[200];
	BOOLEAN		read;

	read = ReadCfgFile(&File);

	StringToUnicode("gammu",x);

	StringToUnicode("port",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->Device));

	StringToUnicode("connection",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->Protocol));

	StringToUnicode("model",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->Phone));

	StringToUnicode("compatibility",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->Compatibility));

	StringToUnicode("logformat",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->LogLevel));

	StringToUnicode("logfile",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->LogFile));

	StringToUnicode("synchronizetime",x2);
	File.SetValue(x,x2,StringToUnicodeReturn(CFG->SyncTime));

	if (read) {
		File.SaveToDisk(UnicodeToStringReturn(File.Info.ID.data()));
	} else {
		File.SaveToDisk("gammurc");
	}
}

/**
 * method for setting handler, which can handle all frames returned from phones
 */
void GSM_StateMachine::SetUserReply(GSM_Error(*UsrReply)(int MsgLength, unsigned char MsgType, unsigned char *MsgBuffer, void *Struct, int RequestID))
{
        if (Phones->Current != NULL) Phones->Current->SetUserReply(UsrReply);
}

void GSM_StateMachine::SetSMSSendReply(void(*SMSSndReply)(int TPMR))
{
        if (Phones->Current != NULL) Phones->Current->SetSMSSendReply(SMSSndReply);
}
