/* (C) 2005-2007 by Marcin Wiacek www.mwiacek.com */

#include <string.h>

#include "../../cfg/config.h"
#include "../misc/coding/coding.h"
#include "gsmfiles.h"
#include "gsmmisc.h"

typedef struct {
        GSM_Error_Code        ErrorNum;
        char                  *ErrorText;
} PrintErrorEntry;

char *GSM_GetErrorInfo(GSM_Error e)
{
        PrintErrorEntry PrintErrorEntries[] = {
		{GSM_ERR_DRIVER_NOT_AVAILABLE,		"Some required driver or operating system part not available."},
		{GSM_ERR_DEVICE_WRITE,			"Error writing device"},
		{GSM_ERR_DEVICE_READ,			"Error reading device"},
		{GSM_ERR_DEVICE_OPEN,			"Error opening device"},
		{GSM_ERR_DEVICE_CLOSE,			"Error closing device"},
                {GSM_ERR_NONE,                          "No error."},
                {GSM_ERR_UNKNOWN,                       "Unknown error. Please report."},
                {GSM_ERR_SOURCE_NOT_COMPILED,           "Some parts of source not compiled."},
                {GSM_ERR_PROTOCOL_STRING_UNKNOWN,       "Protocol specified in config is not known."},
                {GSM_ERR_PHONE_STRING_UNKNOWN,          "Phone model specified in config is not known."},
                {GSM_ERR_OTHER_PROTOCOL,                "Phone model specified in config works with others protocols only."},
		{GSM_ERR_TIMEOUT,			"No response in specified time."},
		{GSM_ERR_EMPTY,				"Empty"},
		{GSM_ERR_NOT_SUPPORTED,			"Not supported by phone"},
		{GSM_ERR_INSIDE_PHONE_MENU,		"Data are edited in phone menu. Leave it before editing from PC."},
		{GSM_ERR_NO_SIM,			"Function requires SIM card"},
		{GSM_ERR_GPLUS_NOT_SUPPORTED,		"Not supported by Gammu+"},
//	GSM_ERR_MEMORY,
//	GSM_ERR_FOLDER_PART,
		{GSM_ERR_FILE_CHECKSUM,			"File checksum error"},
		{GSM_ERR_MUST_BE_FILE,			"%s must be file, not folder"},
		{GSM_ERR_MUST_BE_FOLDER,		"%s must be folder, not file"},
		{GSM_ERR_FILE_EXIST,			"There is already file %s"},
		{GSM_ERR_FOLDER_EXIST,			"There is already folder %s"},
		{GSM_ERR_FOLDER_NOT_EMPTY,		"Folder %s is not empty"},
		{GSM_ERR_FILE_NOT_EXIST,		"File %s doesn't exist (folder with this name too)"},
		{GSM_ERR_FILE_FOLDER_NOT_EXIST,		"File and folder %s doesn't exist"},
		{GSM_ERR_FOLDER_NOT_EXIST,		"Folder %s doesn't exist (file with this name too)"},
		{GSM_ERR_FILE_TOO_SHORT,		"File is too short"},

                {GSM_ERR_NONE,                          ""}
        };
        char   		*def;
	static char 	buff[500];
        int    		i = 0;

        while (1) {
                if (PrintErrorEntries[i].ErrorText[0] == 0x00) {
			def = "Unknown error.";
		        return def;
		} else if (PrintErrorEntries[i].ErrorNum == e.Code) {
                        def = PrintErrorEntries[i].ErrorText;
			if (strstr(def,"%s")!=NULL) {
				sprintf(buff,def,UnicodeToStringReturn(e.Parameter.data()));
				return buff;
			} else {
				return def;
			}
                }
                i++;
        }
}

GSM_Error GSM_Return_Error(GSM_Error_Code Code) 
{
	GSM_Error error;

	error.Code=Code;
	return error;
}

typedef struct {
	char 		*Description;
	GSM_MemoryType	Type;
} GSM_MemoryTypeInfo;

GSM_MemoryTypeInfo GSM_MemoryTypeInfos[] = {
	"ME", 		MEM_PHONE,
	"DC", 		MEM_PHONE_DIALLED,
	"MC", 		MEM_PHONE_MISSED,
	"RC", 		MEM_PHONE_RECEIVED,
	"SM", 		MEM_SIM,
	"ON", 		MEM_SIM_OWN,
	"MT",		MEM_SIM_PHONE,
	"unknown", 	MEM_Unknown
};

GSM_MemoryType GSM_GetMemoryType(char *Buffer)
{
	int i=0;
	
	while(GSM_MemoryTypeInfos[i].Type != MEM_Unknown) {
		if (!strcmp(GSM_MemoryTypeInfos[i].Description,Buffer)) return GSM_MemoryTypeInfos[i].Type;
		i++;
	}
	return MEM_Unknown;
}

char *GSM_GetMemoryName(GSM_MemoryType Type)
{
	int i=0;
	
	while(1) {
		if (GSM_MemoryTypeInfos[i].Type == Type) return GSM_MemoryTypeInfos[i].Description;
		i++;
	}
}

GSM_Error GSM_DecodeSMSNumber(wchar_t *Destination, const unsigned char *Source, int len, BOOLEAN semioctets)
{
	int len2 = Source[0],pos=0;

	if (len==0) {
		Destination[0]=0;
		return GSM_Return_Error(GSM_ERR_NONE);
	}
	
	if (semioctets) {
		if (len2 % 2) len2++;
		len2 = len2/2+1;
	}
	len2--;

	if ((Source[1] & 112) == (0xD0 & 112)) {		
		//alphanumeric
		len2+=4;
		GSM_DecodeSMSText(Destination, Source+2, Source[0], Source[0], 0);
		return GSM_Return_Error(GSM_ERR_NONE);
	}
	if ((Source[1] & 112) == (0x91 & 112)) Destination[pos++] = '+';
	BCDToUnicode(Source+2,Destination+pos,len2);
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_EncodeSMSNumber(unsignedstring *Destination, wchar_t *Source, BOOLEAN semioctets)
{
        unsigned char 	buff[50],type=0x91;
	int 		len2,i,len3;
	GSM_Error	error;

	if (UnicodeLength(Source) == 0) return GSM_Return_Error(GSM_ERR_NONE);
	for (i=0;i<UnicodeLength(Source);i++) {
		if (i==0 && Source[0] == '+') continue;
		if (!isdigit(UnicodeToStringReturn(Source)[i])) {
			error = GSM_EncodeSMSText(buff, Source, &len2, 50, 0);
			if (error.Code != GSM_ERR_NONE) return error;
			len3 = len2;
			type = 0xD0;
			break;
		}
	}

	if (type == 0x91) {
		if (Source[0] == '+') {
			len2 = StringToBCD((const unsigned char *)UnicodeToStringReturn(Source+1), buff, UnicodeLength(Source)-1);
			len3 = UnicodeLength(Source)-1;
		} else {
			len2 = StringToBCD((const unsigned char *)UnicodeToStringReturn(Source), buff, UnicodeLength(Source));
			len3 = UnicodeLength(Source);
			type = 0x81;
		}
	}

	if (!semioctets) len3=len2+1;

	Destination->push_back(len3);
	Destination->push_back(type);
	Destination->append(buff,len2);
	return GSM_Return_Error(GSM_ERR_NONE);
}

/* ETSI GSM 03.38, section 6.2.1: Default alphabet for SMS messages */
static unsigned char GSM_DefaultAlphabetUnicode[128+1][2] =
{
	{0x00,0x40},{0x00,0xa3},{0x00,0x24},{0x00,0xA5},
	{0x00,0xE8},{0x00,0xE9},{0x00,0xF9},{0x00,0xEC},/*0x08*/
	{0x00,0xF2},{0x00,0xC7},{0x00,'\n'},{0x00,0xD8},
	{0x00,0xF8},{0x00,'\r'},{0x00,0xC5},{0x00,0xE5},
	{0x03,0x94},{0x00,0x5f},{0x03,0xA6},{0x03,0x93},
	{0x03,0x9B},{0x03,0xA9},{0x03,0xA0},{0x03,0xA8},
	{0x03,0xA3},{0x03,0x98},{0x03,0x9E},{0x00,0xb9},
	{0x00,0xC6},{0x00,0xE6},{0x00,0xDF},{0x00,0xC9},/*0x20*/
	{0x00,' ' },{0x00,'!' },{0x00,'\"'},{0x00,'#' },
	{0x00,0xA4},{0x00,'%' },{0x00,'&' },{0x00,'\''},
	{0x00,'(' },{0x00,')' },{0x00,'*' },{0x00,'+' },
	{0x00,',' },{0x00,'-' },{0x00,'.' },{0x00,'/' },/*0x30*/
	{0x00,'0' },{0x00,'1' },{0x00,'2' },{0x00,'3' },
	{0x00,'4' },{0x00,'5' },{0x00,'6' },{0x00,'7' },
	{0x00,'8' },{0x00,'9' },{0x00,':' },{0x00,';' },
	{0x00,'<' },{0x00,'=' },{0x00,'>' },{0x00,'?' },/*0x40*/
	{0x00,0xA1},{0x00,'A' },{0x00,'B' },{0x00,'C' },
	{0x00,'D' },{0x00,'E' },{0x00,'F' },{0x00,'G' },
	{0x00,'H' },{0x00,'I' },{0x00,'J' },{0x00,'K' },
	{0x00,'L' },{0x00,'M' },{0x00,'N' },{0x00,'O' },
	{0x00,'P' },{0x00,'Q' },{0x00,'R' },{0x00,'S' },
	{0x00,'T' },{0x00,'U' },{0x00,'V' },{0x00,'W' },
	{0x00,'X' },{0x00,'Y' },{0x00,'Z' },{0x00,0xC4},
	{0x00,0xD6},{0x00,0xD1},{0x00,0xDC},{0x00,0xA7},
	{0x00,0xBF},{0x00,'a' },{0x00,'b' },{0x00,'c' },
	{0x00,'d' },{0x00,'e' },{0x00,'f' },{0x00,'g' },
	{0x00,'h' },{0x00,'i' },{0x00,'j' },{0x00,'k' },
	{0x00,'l' },{0x00,'m' },{0x00,'n' },{0x00,'o' },
	{0x00,'p' },{0x00,'q' },{0x00,'r' },{0x00,'s' },
	{0x00,'t' },{0x00,'u' },{0x00,'v' },{0x00,'w' },
	{0x00,'x' },{0x00,'y' },{0x00,'z' },{0x00,0xE4},
	{0x00,0xF6},{0x00,0xF1},{0x00,0xFC},{0x00,0xE0},
	{0x00,0x00}
};

/* ETSI GSM 3.38
 * Some sequences of 2 default alphabet chars (for example,
 * 0x1b, 0x65) are visible as one single additional char (for example,
 * 0x1b, 0x65 gives Euro char saved in Unicode as 0x20, 0xAC)
 * This table contains:
 * 1. two first chars means sequence of chars from GSM default alphabet
 * 2. two second is target (encoded) char saved in Unicode
 */
static unsigned char GSM_DefaultAlphabetCharsExtension[][4] =
{
	{0x1b,0x14,0x00,0x5e},	/* ^	*/
	{0x1b,0x28,0x00,0x7b},	/* {	*/
	{0x1b,0x29,0x00,0x7d},	/* }	*/
	{0x1b,0x2f,0x00,0x5c},	/* \	*/
	{0x1b,0x3c,0x00,0x5b},	/* [	*/
	{0x1b,0x3d,0x00,0x7E},	/* ~	*/
	{0x1b,0x3e,0x00,0x5d},	/* ]	*/
	{0x1b,0x40,0x00,0x7C},	/* |	*/
	{0x1b,0x65,0x20,0xAC},	/* Euro */
	{0x00,0x00,0x00,0x00}
};

GSM_Error GSM_DecodeSMSText(wchar_t *Destination, const unsigned char *Source, int len, int len3, int udhlen)
{
	unsigned int		i,chr,j,z,p,q,r,len2;
	wchar_t			prev;

	i = 0;
	do {
		i+=7;
	} while ((i-udhlen)%i<0);
	len2 = len-(udhlen*8 + (i-udhlen)%i) / 7;
	j = 1;
	for (i=1;i<(unsigned int)udhlen*7;i++) {
		if (j == 128) {
			j = 1;
		} else {
			j = j * 2;
		}
	}
	i = 0;
	q = 0;
	for (p=0;p<len2;p++) {
		chr = 0;
		for (z=1;z<128;z=z*2) {
			if (Source[i]&j) chr += z;
			if (j == 128) {
				j = 1;
				i++;
				if (i <= (unsigned int)len3) continue;
				return GSM_Return_Error(GSM_ERR_GPLUS_NOT_SUPPORTED);
			}
			j = j * 2;
		}
		Destination[q] = GSM_DefaultAlphabetUnicode[chr][0]*256+GSM_DefaultAlphabetUnicode[chr][1];
		if (p>0) {
			r = 0;
			while (GSM_DefaultAlphabetCharsExtension[r][0] != 0x00) {
				if (prev == GSM_DefaultAlphabetCharsExtension[r][0] &&
				    chr == GSM_DefaultAlphabetCharsExtension[r][1]) {
					q--;
					Destination[q] = GSM_DefaultAlphabetCharsExtension[r][2]*256+
						  GSM_DefaultAlphabetCharsExtension[r][3];
					break;
				}
				r++;
			}
		}
		prev = chr;
		q++;
	}
	Destination[q] = 0;

	return GSM_Return_Error(GSM_ERR_NONE);
}

/* There are many national chars with "adds". In phone they're normally
 * changed to "plain" Latin chars. We have such functionality too.
 * This table is automatically created from convert.txt file (see
 * /docs/developers) using --makeconverttable. It contains such chars
 * to replace in order:
 * 1. original char (Unicode) 2. destination char (Unicode)
 */
static unsigned char ConvertTable[] =
"\x00\xc0\x00\x41\x00\xe0\x00\x61\x00\xc1\x00\x41\x00\xe1\x00\x61\x00\xc2\x00\x41\x00\xe2\x00\x61\x00\xc3\x00\x41\x00\xe3\x00\x61\x1e\xa0\x00\x41\x1e\xa1\x00\x61\x1e\xa2\x00\x41\x1e\xa3\x00\x61\x1e\xa4\x00\x41\x1e\xa5\x00\x61\x1e\xa6\x00\x41\x1e\xa7\x00\x61\x1e\xa8\x00\x41\x1e\xa9\x00\x61\x1e\xaa\x00\x41\x1e\xab\x00\x61\x1e\xac\x00\x41\x1e\xad\x00\x61\x1e\xae\x00\x41\x1e\xaf\x00\x61\x1e\xb0\x00\x41\x1e\xb1\x00\x61\x1e\xb2\x00\x41\x1e\xb3\x00\x61\x1e\xb4\x00\x41\x1e\xb5\x00\x61\x1e\xb6\x00\x41\x1e\xb7\x00\x61\x01\xcd\x00\x41\x01\xce\x00\x61\x01\x00\x00\x41\x01\x01\x00\x61\x01\x02\x00\x41\x01\x03\x00\x61\x01\x04\x00\x41\x01\x05\x00\x61\x01\xfb\x00\x61\x01\x06\x00\x43\x01\x07\x00\x63\x01\x08\x00\x43\x01\x09\x00\x63\x01\x0a\x00\x43\x01\x0b\x00\x63\x01\x0c\x00\x43\x01\x0d\x00\x63\x00\xe7"\
"\x00\x63\x01\x0e\x00\x44\x01\x0f\x00\x64\x01\x10\x00\x44\x01\x11\x00\x64\x00\xc8\x00\x45\x00\xca\x00\x45\x00\xea\x00\x65\x00\xcb\x00\x45\x00\xeb\x00\x65\x1e\xb8\x00\x45\x1e\xb9\x00\x65\x1e\xba\x00\x45\x1e\xbb\x00\x65\x1e\xbc\x00\x45\x1e\xbd\x00\x65\x1e\xbe\x00\x45\x1e\xbf\x00\x65\x1e\xc0\x00\x45\x1e\xc1\x00\x65\x1e\xc2\x00\x45\x1e\xc3\x00\x65\x1e\xc4\x00\x45\x1e\xc5\x00\x65\x1e\xc6\x00\x45\x1e\xc7\x00\x65\x01\x12\x00\x45\x01\x13\x00\x65\x01\x14\x00\x45\x01\x15\x00\x65\x01\x16\x00\x45\x01\x17\x00\x65\x01\x18\x00\x45\x01\x19\x00\x65\x01\x1a\x00\x45\x01\x1b\x00\x65\x01\x1c\x00\x47\x01\x1d\x00\x67\x01\x1e\x00\x47\x01\x1f\x00\x67\x01\x20\x00\x47\x01\x21\x00\x67\x01\x22\x00\x47\x01\x23\x00\x67\x01\x24\x00\x48\x01\x25\x00\x68\x01\x26\x00\x48\x01\x27\x00\x68\x00\xcc\x00\x49\x00\xcd\x00\x49\x00\xed"\
"\x00\x69\x00\xce\x00\x49\x00\xee\x00\x69\x00\xcf\x00\x49\x00\xef\x00\x69\x01\x28\x00\x49\x01\x29\x00\x69\x01\x2a\x00\x49\x01\x2b\x00\x69\x01\x2c\x00\x49\x01\x2d\x00\x69\x01\x2e\x00\x49\x01\x2f\x00\x69\x01\x30\x00\x49\x01\x31\x00\x69\x01\xcf\x00\x49\x01\xd0\x00\x69\x1e\xc8\x00\x49\x1e\xc9\x00\x69\x1e\xca\x00\x49\x1e\xcb\x00\x69\x01\x34\x00\x4a\x01\x35\x00\x6a\x01\x36\x00\x4b\x01\x37\x00\x6b\x01\x39\x00\x4c\x01\x3a\x00\x6c\x01\x3b\x00\x4c\x01\x3c\x00\x6c\x01\x3d\x00\x4c\x01\x3e\x00\x6c\x01\x3f\x00\x4c\x01\x40\x00\x6c\x01\x41\x00\x4c\x01\x42\x00\x6c\x01\x43\x00\x4e\x01\x44\x00\x6e\x01\x45\x00\x4e\x01\x46\x00\x6e\x01\x47\x00\x4e\x01\x48\x00\x6e\x01\x49\x00\x6e\x00\xd2\x00\x4f\x00\xd3\x00\x4f\x00\xf3\x00\x6f\x00\xd4\x00\x4f\x00\xf4\x00\x6f\x00\xd5\x00\x4f\x00\xf5\x00\x6f\x01\x4c\x00\x4f\x01\x4d"\
"\x00\x6f\x01\x4e\x00\x4f\x01\x4f\x00\x6f\x01\x50\x00\x4f\x01\x51\x00\x6f\x01\xa0\x00\x4f\x01\xa1\x00\x6f\x01\xd1\x00\x4f\x01\xd2\x00\x6f\x1e\xcc\x00\x4f\x1e\xcd\x00\x6f\x1e\xce\x00\x4f\x1e\xcf\x00\x6f\x1e\xd0\x00\x4f\x1e\xd1\x00\x6f\x1e\xd2\x00\x4f\x1e\xd3\x00\x6f\x1e\xd4\x00\x4f\x1e\xd5\x00\x6f\x1e\xd6\x00\x4f\x1e\xd7\x00\x6f\x1e\xd8\x00\x4f\x1e\xd9\x00\x6f\x1e\xda\x00\x4f\x1e\xdb\x00\x6f\x1e\xdc\x00\x4f\x1e\xdd\x00\x6f\x1e\xde\x00\x4f\x1e\xdf\x00\x6f\x1e\xe0\x00\x4f\x1e\xe1\x00\x6f\x1e\xe2\x00\x4f\x1e\xe3\x00\x6f\x01\x54\x00\x52\x01\x55\x00\x72\x01\x56\x00\x52\x01\x57\x00\x72\x01\x58\x00\x52\x01\x59\x00\x72\x01\x5a\x00\x53\x01\x5b\x00\x73\x01\x5c\x00\x53\x01\x5d\x00\x73\x01\x5e\x00\x53\x01\x5f\x00\x73\x01\x60\x00\x53\x01\x61\x00\x73\x01\x62\x00\x54\x01\x63\x00\x74\x01\x64\x00\x54\x01\x65"\
"\x00\x74\x01\x66\x00\x54\x01\x67\x00\x74\x00\xd9\x00\x55\x00\xda\x00\x55\x00\xfa\x00\x75\x00\xdb\x00\x55\x00\xfb\x00\x75\x01\x68\x00\x55\x01\x69\x00\x75\x01\x6a\x00\x55\x01\x6b\x00\x75\x01\x6c\x00\x55\x01\x6d\x00\x75\x01\x6e\x00\x55\x01\x6f\x00\x75\x01\x70\x00\x55\x01\x71\x00\x75\x01\x72\x00\x55\x01\x73\x00\x75\x01\xaf\x00\x55\x01\xb0\x00\x75\x01\xd3\x00\x55\x01\xd4\x00\x75\x01\xd5\x00\x55\x01\xd6\x00\x75\x01\xd7\x00\x55\x01\xd8\x00\x75\x01\xd9\x00\x55\x01\xda\x00\x75\x01\xdb\x00\x55\x01\xdc\x00\x75\x1e\xe4\x00\x55\x1e\xe5\x00\x75\x1e\xe6\x00\x55\x1e\xe7\x00\x75\x1e\xe8\x00\x55\x1e\xe9\x00\x75\x1e\xea\x00\x55\x1e\xeb\x00\x75\x1e\xec\x00\x55\x1e\xed\x00\x75\x1e\xee\x00\x55\x1e\xef\x00\x75\x1e\xf0\x00\x55\x1e\xf1\x00\x75\x01\x74\x00\x57\x01\x75\x00\x77\x1e\x80\x00\x57\x1e\x81\x00\x77\x1e\x82"\
"\x00\x57\x1e\x83\x00\x77\x1e\x84\x00\x57\x1e\x85\x00\x77\x00\xdd\x00\x59\x00\xfd\x00\x79\x00\xff\x00\x79\x01\x76\x00\x59\x01\x77\x00\x79\x01\x78\x00\x59\x1e\xf2\x00\x59\x1e\xf3\x00\x75\x1e\xf4\x00\x59\x1e\xf5\x00\x79\x1e\xf6\x00\x59\x1e\xf7\x00\x79\x1e\xf8\x00\x59\x1e\xf9\x00\x79\x01\x79\x00\x5a\x01\x7a\x00\x7a\x01\x7b\x00\x5a\x01\x7c\x00\x7a\x01\x7d\x00\x5a\x01\x7e\x00\x7a\x01\xfc\x00\xc6\x01\xfd\x00\xe6\x01\xfe\x00\xd8\x01\xff\x00\xf8\x00\x00";

GSM_Error GSM_EncodeSMSText(unsigned char *Destination, const wchar_t *Source, int *len, int maxlen, int udhlen)
{
	int		pos=0,j,destbit,z,gsmcharlen,l=0;
	unsigned char 	gsmchar[2];
	BOOLEAN 	found;
	wchar_t 	srcchar;

	(*len) 			= 0;
	Destination[l] 	= 0;
	destbit			= 1; //fixme
	while (Source[pos] != 0) {
		found 		= FALSE;
		gsmcharlen 	= 1;

		j = 0;
		while (GSM_DefaultAlphabetCharsExtension[j][0]*256+GSM_DefaultAlphabetCharsExtension[j][1]!=0) {
			if (GSM_DefaultAlphabetCharsExtension[j][2]*256+GSM_DefaultAlphabetCharsExtension[j][3] == Source[pos]) {
				gsmchar[0] = GSM_DefaultAlphabetCharsExtension[j][0];
				gsmchar[1] = GSM_DefaultAlphabetCharsExtension[j][1];
				gsmcharlen = 2;
				found = TRUE;
				break;
			}
			j++;
		}

		if (found==FALSE) {
			srcchar = Source[pos];
			j = 0;
			while (ConvertTable[j*4]*256+ConvertTable[j*4+1]!=0) {
				if (Source[pos] == ConvertTable[j*4]*256+ConvertTable[j*4+1]) {
					srcchar = ConvertTable[j*4+2]*256+ConvertTable[j*4+3];
					break;
				}
				j++;
			}

			j = 0;
			while (GSM_DefaultAlphabetUnicode[j][0]*256+GSM_DefaultAlphabetUnicode[j][1]!=0) {
				if (GSM_DefaultAlphabetUnicode[j][0]*256+GSM_DefaultAlphabetUnicode[j][1] == Source[pos]) {
					gsmchar[0] = j;
					found = TRUE;
					break;
				}
				if (GSM_DefaultAlphabetUnicode[j][0]*256+GSM_DefaultAlphabetUnicode[j][1] == srcchar) {
					gsmchar[0] = j;
					found = TRUE;
					break;
				}
				j++;
			}
		}

		if (found==FALSE) gsmchar[0] = '?';

		for (j=0;j<gsmcharlen;j++) {
			for (z=1;z<128;z*=2) {
				if (gsmchar[j] & z) Destination[l] += destbit;
				if (destbit==128) {
					l++;
					if (l > maxlen) return GSM_Return_Error(GSM_ERR_UNKNOWN);
					destbit = 1;
					Destination[l] = 0;
					continue;
				}
				destbit*=2;
			}
		}
		(*len)+=gsmcharlen;
		pos++;
	}
	l++;
	return GSM_Return_Error(GSM_ERR_NONE);
}

//-----------------------------------------------------------------------------

GSM_SMSMMSFoldersSubEntry::GSM_SMSMMSFoldersSubEntry()
{
	Name = NULL;
	Next = NULL;
}

GSM_SMSMMSFoldersSubEntry::~GSM_SMSMMSFoldersSubEntry()
{
	delete(Next);
}

void GSM_SMSMMSFoldersSubEntry::SetName(wchar_t *N)
{
	int len = (UnicodeLength(N)+1)*sizeof(wchar_t);
	
	Name = (wchar_t *)realloc(Name,len);
	memcpy(Name,N,len);
}

wchar_t *GSM_SMSMMSFoldersSubEntry::GetName()
{
	if (Name == NULL) return 0x00;
	return Name;
}

void GSM_SMSMMSFoldersSubEntry::SetNext(GSM_SMSMMSFoldersSubEntry *Nxt)
{
	Next = Nxt;
}

GSM_SMSMMSFoldersSubEntry *GSM_SMSMMSFoldersSubEntry::GetNext()
{
	return Next;
}

/* ------------------------------------------------------------------------ */

GSM_SMSMMSFolders::GSM_SMSMMSFolders()
{
	Entries = NULL;
}

GSM_SMSMMSFolders::~GSM_SMSMMSFolders()
{
	delete(Entries);
}

BOOLEAN GSM_SMSMMSFolders::GetNext(GSM_SMSMMSFoldersSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

void GSM_SMSMMSFolders::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}

GSM_Error GSM_SMSMMSFolders::Add(wchar_t *Name, BOOLEAN FF, BOOLEAN Inbox, BOOLEAN MMS, GSM_MemoryType mem, wchart *ID, int Num, int OtherNum)
{
	GSM_SMSMMSFoldersSubEntry *Entry,*Entry2;

	Entry 		= new GSM_SMSMMSFoldersSubEntry;
	Entry->Inbox  	= Inbox;
	Entry->MMS	= MMS;
	Entry->Memory 	= mem;
	Entry->FileSystem = FF;
	Entry->ID.append(ID->data());
	Entry->SetName(Name);
	Entry->Number = Num;
	Entry->OtherNumber = OtherNum;

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

static int GetRSSVersion(char *Buffer)
{
	int retval = 0, pos = 0;

	retval = atoi(Buffer) * 10000;
	while (Buffer[pos] != '.') {
		pos++;
		if (pos == strlen(Buffer)) return retval;
	}
	pos++;
	retval += atoi(Buffer+pos) * 100;
	while (Buffer[pos] != '.') {
		pos++;
		if (pos == strlen(Buffer)) return retval;
	}
	pos++;
	return retval + atoi(Buffer+pos);
}

GSM_Error CheckRSS(unsignedstring *StableStr, unsignedstring *TestStr)
{
	GSM_File 	File;
	GSM_Error	error;
	unsigned int	pos=0,oldpos=0,i;
	char		buff[200];
	unsignedstring	X;
	
	X.push_back(0x00);

	StableStr->clear();
	TestStr->clear();

	if (!File.ReadFromWWW("www.mwiacek.com","gsm/soft/gplus.rss")) return GSM_Return_Error(GSM_ERR_UNKNOWN);
	
	while (pos < File.Buffer.size()) {
		if (File.Buffer.data()[pos] != 10) {
			pos++;
			continue;
		}
		File.Buffer.replace(pos,1,X);
		if (strstr((char *)File.Buffer.data()+oldpos,"<title>") == NULL ||
		    strstr((char *)File.Buffer.data()+oldpos,"</title>")== NULL ||
		    strstr((char *)File.Buffer.data()+oldpos,"win32")   != NULL) {
			pos++;
			oldpos = pos;
			continue;
		}
		if (StableStr->length() == 0 && strstr((char *)File.Buffer.data()+oldpos,"stable version")!=NULL) {
			sprintf(buff,strstr((char *)File.Buffer.data()+oldpos,"stable version")+15);
			for (i=0;i<strlen(buff);i++) {
				if (buff[i] == '<') {
					buff[i] = 0;
					break;
				}
			}
			if (GetRSSVersion(buff)>GetRSSVersion(VERSION)) {
				StableStr->append((const unsigned char *)buff,strlen(buff));				
			}
		}
		if (TestStr->length() == 0 && strstr((char *)File.Buffer.data()+oldpos,"test version")!=NULL) {
			sprintf(buff,strstr((char *)File.Buffer.data()+oldpos,"test version")+13);
			for (i=0;i<strlen(buff);i++) {
				if (buff[i] == '<') {
					buff[i] = 0;
					break;
				}
			}
			if (GetRSSVersion(buff)>GetRSSVersion(VERSION)) {
				TestStr->append((const unsigned char *)buff,strlen(buff));				
			}
		}
		pos++;
		oldpos = pos;
		if (StableStr->length() != 0 && TestStr->length() != 0) break;
	}

	if (StableStr->length() != 0 && TestStr->length() != 0 &&
	    GetRSSVersion((char *)StableStr->data()) > GetRSSVersion((char *)TestStr->data())) {
		TestStr->clear();
	}

        return GSM_Return_Error(GSM_ERR_NONE);
}
