/* (C) 2006-2007 by Marcin Wiacek www.mwiacek.com */

#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>
#ifdef WIN32
#  include <sys/types.h>
#  ifndef BORLANDC
#    include <sys/utime.h>
#  endif
#else
#  include <netdb.h>
#  include <netinet/in.h>
#  include <utime.h>
#endif

#include "../../cfg/config.h"
#include "../misc/coding/coding.h"
#include "gsmfiles.h"

GSM_FileFolderInfo::GSM_FileFolderInfo()
{
	Name = NULL;
	ID   = NULL;
}

GSM_FileFolderInfo::~GSM_FileFolderInfo()
{
	free(Name);
	free(ID);
}

GSM_Error GSM_FileFolderInfo::SetName(wchar_t *Txt)
{
	int len = (UnicodeLength(Txt)+1)*sizeof(wchar_t);

	Name = (wchar_t *)realloc(Name,len);
	memcpy(Name,Txt,len);

	return GSM_Return_Error(GSM_ERR_NONE);
}

wchar_t *GSM_FileFolderInfo::GetName()
{
	return Name;
}

GSM_Error GSM_FileFolderInfo::SetID(wchar_t *Txt)
{
	int len = (UnicodeLength(Txt)+1)*sizeof(wchar_t);

	ID = (wchar_t *)realloc(ID,len);
	memcpy(ID,Txt,len);

	return GSM_Return_Error(GSM_ERR_NONE);
}

wchar_t *GSM_FileFolderInfo::GetID()
{
	return ID;
}

//-----------------------------------------------------------------------------

GSM_FileFolderInfoListSubEntry::GSM_FileFolderInfoListSubEntry()
{
	Next = NULL;
}

GSM_FileFolderInfoListSubEntry::~GSM_FileFolderInfoListSubEntry()
{
	delete(Next);
}

GSM_FileFolderInfoListSubEntry *GSM_FileFolderInfoListSubEntry::GetNext()
{
	return Next;
}

void GSM_FileFolderInfoListSubEntry::SetNext(GSM_FileFolderInfoListSubEntry *Nxt)
{
	Next = Nxt;
}

//-----------------------------------------------------------------------------

GSM_FileFolderInfoList::GSM_FileFolderInfoList()
{
	Entries = NULL;
}

GSM_FileFolderInfoList::~GSM_FileFolderInfoList()
{
	delete (Entries);
}

GSM_Error GSM_FileFolderInfoList::AddSubEntry(GSM_FileFolderInfo *FInfo)
{
	GSM_FileFolderInfoListSubEntry *Entry,*Entry2;

	Entry = new GSM_FileFolderInfoListSubEntry;
	if (Entry==NULL) return GSM_Return_Error(GSM_ERR_UNKNOWN);

	Entry->Info.Folder 		= FInfo->Folder;
	Entry->Info.DRMForwardLock 	= FInfo->DRMForwardLock;
	Entry->Info.ReadOnly 		= FInfo->ReadOnly;
	Entry->Info.Hidden 		= FInfo->Hidden;
	Entry->Info.System 		= FInfo->System;
	Entry->Info.ModificationDateTimeAvailable = FInfo->ModificationDateTimeAvailable;
	if (FInfo->ModificationDateTimeAvailable == TRUE) {
		memcpy(&Entry->Info.ModificationDateTime,&FInfo->ModificationDateTime,sizeof(GSM_DateTime));
	}
	Entry->Info.Size = FInfo->Size;
	Entry->Info.SetID(FInfo->GetID());
	if (FInfo->GetName()!=NULL) Entry->Info.SetName(FInfo->GetName());

	if (Entries == NULL) {
		Entries = Entry;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(Entry);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

BOOLEAN GSM_FileFolderInfoList::GetNext(GSM_FileFolderInfoListSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

void GSM_FileFolderInfoList::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}

//----------------------------------------------------------------------------------

GSM_File::GSM_File()
{
}

GSM_File::~GSM_File()
{
}

GSM_Error GSM_File::ReadFromDisk(wchar_t *FileName)
{
	int 		i = 1000;
	FILE		*file;
	unsigned char	buffer[1000];
	GSM_Error	error;
	GSM_DateTime	dt;
	struct stat     fileinfo;

	if (FileName[0] == 0x00) return GSM_Return_Error(GSM_ERR_UNKNOWN);
	file = fopen(UnicodeToStringReturn(FileName),"rb");
	if (file == NULL) {
		error.Code=GSM_ERR_FILE_NOT_EXIST;
		error.Parameter.append((const wchar_t *)FileName,UnicodeLength(FileName));
		return error;
	}

	Buffer.clear();
	while (i == 1000) {
		i = fread(buffer,1,1000,file);
		Buffer.append((const unsigned char *)buffer,i);		
	}

	fclose(file);
 	Info.Size = Buffer.size();

	Info.ModificationDateTimeAvailable=FALSE;
	if (stat(UnicodeToStringReturn(FileName),&fileinfo) == 0) {
		Info.ModificationDateTimeAvailable=TRUE;
		dt = TimeT2GSMDateTime(&fileinfo.st_mtime);
		memcpy(&Info.ModificationDateTime,&dt,sizeof(GSM_DateTime));
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_File::SaveToDisk(char *FileName)
{
	FILE		*file;
#ifndef BORLANDC
	struct utimbuf	filedate;
#endif

	if (FileName[0] == 0x00) return GSM_Return_Error(GSM_ERR_UNKNOWN);
	file = fopen(FileName,"wb");
	if (file == NULL) return GSM_Return_Error(GSM_ERR_UNKNOWN);
	fwrite(Buffer.data(),1,Buffer.size(),file);
	fclose(file);
#ifndef BORLANDC
	if (Info.ModificationDateTimeAvailable) {
		/* access time */
		filedate.actime  = GSMDateTime2TimeT(&Info.ModificationDateTime);
		/* modification time */
		filedate.modtime = GSMDateTime2TimeT(&Info.ModificationDateTime);
		utime(FileName,&filedate);
	}
#endif
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_File::ReadFromWWW(char *server, char *filename)
{
	int			s, len;
	struct sockaddr_in 	address;
	struct hostent 		*address2;
	char 			buff[200];
#ifdef WIN32
	WSADATA			wsaData;

	if (WSAStartup(MAKEWORD(1, 1), &wsaData) != 0) return GSM_Return_Error(GSM_ERR_UNKNOWN);
#endif

	s = socket(AF_INET,SOCK_STREAM,0);
#ifdef WIN32
	if (s==INVALID_SOCKET) return GSM_Return_Error(GSM_ERR_UNKNOWN);
#else
	if (s==-1) return GSM_Return_Error(GSM_ERR_UNKNOWN);
#endif

	address2 = gethostbyname(server);
	if (address2 == NULL) return GSM_Return_Error(GSM_ERR_UNKNOWN);

	memset((char *) &address, 0, sizeof(address));
	address.sin_family 	= AF_INET;
	address.sin_port 	= htons(80);
	address.sin_addr.s_addr = *(u_long *) *(address2->h_addr_list);

	if (connect(s,(struct sockaddr *)&address,sizeof(address))<0) {
		return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}

	sprintf(buff,"GET http://%s/%s HTTP/1.0\n\n",server,filename);
	if (send(s,buff,strlen(buff),0)<0) return GSM_Return_Error(GSM_ERR_UNKNOWN);

	Buffer.clear();

#ifdef WIN32
	while ((len=recv(s,buff,200,0))>0) {
#else
	while ((len=read(s,buff,200))>0) {
#endif
		Buffer.append((const unsigned char *)buff,len);
	}
#ifdef WIN32
	closesocket(s);
#else
	close(s);
#endif

	if (Buffer.size()==0) return GSM_Return_Error(GSM_ERR_EMPTY);
	if (strstr((const char *)Buffer.data(),"HTTP/1.1 200 OK")==NULL) {
		Buffer.clear();
		return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}

 	Info.Size = Buffer.size();

	return GSM_Return_Error(GSM_ERR_NONE);
}

void GSM_File::JAD_Find_Line(char *Name, unsignedstring *Value)
{
	char 	Line[2000];
	unsigned int	Pos = 0;

	Value->clear();

	while (Pos<Buffer.size()) {
		Line[0] = 0;
		while (Pos<Buffer.size()) {
			if (Buffer.data()[Pos] == 0x00) break;
			if (Buffer.data()[Pos] == 0x0A || Buffer.data()[Pos] == 0x0D) {
				Pos++;
				if (strlen(Line)!=0x00) break;
				continue;
			}
			Line[strlen(Line) + 1] = 0;
			Line[strlen(Line)]     = Buffer.data()[Pos];
			Pos++;
		}
		if (strlen(Line) == 0) break;
		if (!strncmp(Line,Name,strlen(Name))) {
			Pos = strlen(Name);
			while (Line[Pos] == 0x20) Pos++;
			Value->append((unsigned char *)Line+Pos,strlen(Line+Pos));
			return;
		}
	}
}

void GSM_File::JAD_Find_Data(unsignedstring *Vendor, unsignedstring *Name, unsignedstring *JAR, unsignedstring *Version, int *Size)
{
	unsignedstring Size2;

	JAD_Find_Line("MIDlet-Vendor:", Vendor);
	JAD_Find_Line("MIDlet-Name:", Name);
	JAD_Find_Line("MIDlet-Jar-URL:", JAR);
	JAD_Find_Line("MIDlet-Version:", Version);
	JAD_Find_Line("MIDlet-Jar-Size:", &Size2);
	*Size = -1;
	if (Size2.length() != 0) (*Size) = atoi((char *)Size2.data());
}

void GSM_File::SetWin32EndLines()
{
	unsignedstring str;

	// Changing all #13 or #10 to #13#10 in JAD
	str.push_back(13);
	UnsignedStringReplaceChars(&Buffer, 10, str);
	str.clear();
	str.push_back(13);
	str.push_back(10);
	UnsignedStringReplaceChars(&Buffer, 13, str);
	Info.Size = Buffer.size();

}

//-----------------------------------------------------------------------------

GSM_FileFolderInfoListsSubEntry::GSM_FileFolderInfoListsSubEntry()
{
	Next = NULL;
}

GSM_FileFolderInfoListsSubEntry::~GSM_FileFolderInfoListsSubEntry()
{
	delete(Next);
}

GSM_FileFolderInfoListsSubEntry *GSM_FileFolderInfoListsSubEntry::GetNext()
{
	return Next;
}

void GSM_FileFolderInfoListsSubEntry::SetNext(GSM_FileFolderInfoListsSubEntry *Nxt)
{
	Next = Nxt;
}

//-----------------------------------------------------------------------------

GSM_FileFolderInfoLists::GSM_FileFolderInfoLists()
{
	Entries = NULL;
}

GSM_FileFolderInfoLists::~GSM_FileFolderInfoLists()
{
	delete (Entries);
}

GSM_Error GSM_FileFolderInfoLists::AddSubEntry(GSM_FileFolderInfoListsSubEntry *FInfo)
{
	GSM_FileFolderInfoListsSubEntry *Entry2;

	if (Entries == NULL) {
		Entries = FInfo;
	} else {
		Entry2 = Entries;
		while (Entry2->GetNext()!=NULL) Entry2 = Entry2->GetNext();
		Entry2->SetNext(FInfo);
	}
	return GSM_Return_Error(GSM_ERR_NONE);
}

BOOLEAN GSM_FileFolderInfoLists::GetNext(GSM_FileFolderInfoListsSubEntry **En)
{
	if ((*En) == NULL) {
		(*En) = Entries;
	} else {
		(*En) = (*En)->GetNext();
	}
	if ((*En) == NULL) return FALSE;
	return TRUE;
}

void GSM_FileFolderInfoLists::ClearAll()
{
	delete(Entries);
	Entries = NULL;
}

//----------------------------------------------------------------------------------
