/* (C) 2005-2006 by Marcin Wiacek www.mwiacek.com */

#include "../../../../cfg/config.h" //msvc2005
#include "../../../protocol/gsmprot.h"
#include "../../../misc/coding/coding.h"
#include "../ndct34.h"

GSM_Error GSM_Phone_N6510::GetNextCalendar(GSM_CalendarEntry *Entry, BOOLEAN start, int *Current, int *Max)
{
	if ((*Phones)->Feature("cal62")==TRUE) {
		return DCT34->GetNextCalendar1(Entry,start,Current,Max);
	} else {
		return GetNextCalendar3(Entry,start,Current,Max);
	}
}

GSM_Error GSM_Phone_N6510::ReplyGetCalendarInfo3(GSM_Protocol_Message *msg, DebugInfo **Debug, unsigned char *S)
{
	GSM_Cal_Loc     *CalendarLocations = (GSM_Cal_Loc *)S;
	int	     i;

	(*Debug)->Deb("RECEIVED: calendar entries info method 3\n");

	if (CalendarLocations->Locations.length()==0) {
		CalendarLocations->Length = msg->Buffer.data()[8]*256 + msg->Buffer.data()[9];
		(*Debug)->Deb("   Entries in phone: %i\n",CalendarLocations->Length);
	}

	(*Debug)->Deb("   Entries in frame: %i\n",msg->Buffer.data()[4]);

	for (i=0;i<msg->Buffer.data()[4];i++) {
		CalendarLocations->Locations.push_back(msg->Buffer.data()[12+(i*4)]*256+msg->Buffer.data()[13+(i*4)]);
	}

	if (msg->Buffer.data()[4] == 1 && msg->Buffer.data()[12+(0*4)]*256+msg->Buffer.data()[13+(0*4)] == 0) return GSM_Return_Error(GSM_ERR_EMPTY);
	if (msg->Buffer.data()[4] == 0) return GSM_Return_Error(GSM_ERR_EMPTY);

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Phone_N6510::GetCalendarInfo3()
{
	GSM_Error       error;
	unsigned char   Buff[] = {
		NOKIA_FRAME1, 0x9E, 0xFF, 0xFF, 0x00, 0x00,
		0x00, 0x00,	     // first location number
		0x00};		  // 0 = calendar, 1 = ToDo in 6610 style, 2 = Notes

	CalendarLocations.Locations.clear();
	CalendarLocations.Length = 0;

	while (1) {     
		if (CalendarLocations.Locations.length() > 0) {
			Buff[8] = CalendarLocations.Locations.data()[CalendarLocations.Locations.size()-1] / 256;
			Buff[9] = CalendarLocations.Locations.data()[CalendarLocations.Locations.size()-1] % 256;
		}

		(*Debug)->Deb("SENT: getting locations for calendar method 3\n");
		error = Write(Buff, sizeof(Buff), 0x13, 4, ID_GetCalendarInfo+ID,&CalendarLocations);
		if (error.Code != GSM_ERR_NONE && error.Code != GSM_ERR_EMPTY) return error;

		if (CalendarLocations.Locations.length()==CalendarLocations.Length) {
			return GSM_Return_Error(GSM_ERR_NONE);
		} else {
			if (error.Code == GSM_ERR_EMPTY) {
				CalendarLocations.Length = CalendarLocations.Locations.length();
				return GSM_Return_Error(GSM_ERR_NONE);
			}
		}
	}
}

GSM_Error GSM_Phone_N6510::ReplyGetNextCalendar3(GSM_Protocol_Message *msg, DebugInfo **Debug, unsigned char *S)
{
	time_t			  timet;
	struct tm		       tmstruct;
	GSM_DateTime		    DT,DT2;
	GSM_CalendarEntry	       *Calendar = (GSM_CalendarEntry *)S;
	int			     Alarm,j;
	BOOLEAN			 found = FALSE;
	GSM_Calendar_SubEntryType       AlarmType = Calendar_DateTime_ToneAlarm;

	(*Debug)->Deb("RECEIVED: calendar entry method 3\n");

	switch(msg->Buffer.data()[27]) {
	case 0x00:
		(*Debug)->Deb("   reminder\n");
		Calendar->Type = Calendar_Type_Reminder;
		break;
	case 0x01:
		(*Debug)->Deb("   meeting\n");
		Calendar->Type = Calendar_Type_Meeting;
		break;
	case 0x02:
		(*Debug)->Deb("   call\n");
		Calendar->Type = Calendar_Type_Call;
		break;
	case 0x04:
		(*Debug)->Deb("   birthday\n");
		Calendar->Type = Calendar_Type_Birthday;
		break;
	case 0x08:
		(*Debug)->Deb("   memo\n");
		Calendar->Type = Calendar_Type_Memo;
		break;
	default:
		return GSM_Return_Error(GSM_ERR_UNKNOWN);
	}

	for (j=0;j<5;j++) {
		if (CalendarIcons[j].NoteType == Calendar->Type) {
			found = TRUE;
			break;
		}
	}
	if (found == FALSE) {
		for (j=0;j<5;j++) {
			if (CalendarIcons[j].NoteType == Calendar_Type_Not_Assigned) {
				CalendarIcons[j].NoteType = Calendar->Type;
				CalendarIcons[j].NoteID   = msg->Buffer.data()[21];
				break;
			}
		}	       
	}

	memcpy(&DT,NokiaGetDT(msg->Buffer.data()+28),sizeof(GSM_DateTime));
	/* Garbage seen with 3510i 3.51 */
	if (DT.Month == 0 && DT.Day == 0 && DT.Hour == 0 && DT.Minute == 0) return GSM_Return_Error(GSM_ERR_EMPTY);
	if (Calendar->Type == Calendar_Type_Birthday) {
		DT.Year = msg->Buffer.data()[42]*256 + msg->Buffer.data()[43];
		if (DT.Year == 0xFFFF) DT.Year = 0;
	}
	DT.Second = 0;
	(*Debug)->Deb("   Start date time: %02i-%02i-%04i %02i:%02i:%02i\n",
			DT.Day,DT.Month,DT.Year,DT.Hour,DT.Minute,DT.Second);
	Calendar->AddDateTime(Calendar_DateTime_Start, DT);

	NokiaGetCalendarRecurranceRepeat(*Debug,Calendar, msg->Buffer.data()+40, msg->Buffer.data()+46);

	if(Calendar->Type != Calendar_Type_Birthday) {
		memcpy(&DT2,NokiaGetDT(msg->Buffer.data()+34),sizeof(GSM_DateTime));
		DT2.Second = 0;
		(*Debug)->Deb("   End date time: %02i-%02i-%04i %02i:%02i:%02i\n",
				DT2.Day,DT2.Month,DT2.Year,DT2.Hour,DT2.Minute,DT2.Second);
		Calendar->AddDateTime(Calendar_DateTime_End, DT2);
	}

	if (msg->Buffer.data()[51] != 0) {
		Calendar->AddText(Calendar_Text_Text, NokiaGetUnicodeSimple(msg->Buffer.data()+54,msg->Buffer.data()[51]));
	}

	switch (Calendar->Type) {
	case Calendar_Type_Meeting:
		if (msg->Buffer.data()[52] != 0) {
			Calendar->AddText(Calendar_Text_Location, NokiaGetUnicodeSimple(msg->Buffer.data()+54+msg->Buffer.data()[51]*2,msg->Buffer.data()[52]));
		}
		break;
	case Calendar_Type_Call:
		if (msg->Buffer.data()[52] != 0) {
			Calendar->AddText(Calendar_Text_Phone, NokiaGetUnicodeSimple(msg->Buffer.data()+54+msg->Buffer.data()[51]*2,msg->Buffer.data()[52]));
		}
		break;
	default:
		break;
	}

	Alarm = msg->Buffer.data()[14]*256*256*256+msg->Buffer.data()[15]*256*256;
	Alarm +=msg->Buffer.data()[16]*256+msg->Buffer.data()[17];
	if (Alarm == 0xffffffff) return GSM_Return_Error(GSM_ERR_NONE);

	(*Debug)->Deb("   Alarm difference : %i\n",Alarm);

	if (Calendar->Type == Calendar_Type_Birthday) DT.Year = 2000;
	tmstruct.tm_year	= DT.Year - 1900;
	tmstruct.tm_mon	 = DT.Month - 1;
	tmstruct.tm_mday	= DT.Day;
	tmstruct.tm_hour	= DT.Hour;
	tmstruct.tm_min	 = DT.Minute;
	tmstruct.tm_sec	 = DT.Second;
	tmstruct.tm_isdst       = -1;

	timet = mktime(&tmstruct);
	timet -= 60 * Alarm;

	memcpy(&tmstruct,localtime(&timet),sizeof(struct tm));
	DT.Year	 = tmstruct.tm_year + 1900;
	DT.Month	= tmstruct.tm_mon+1;
	DT.Day	  = tmstruct.tm_mday;
	DT.Hour	 = tmstruct.tm_hour;
	DT.Minute       = tmstruct.tm_min;
	DT.Second       = tmstruct.tm_sec;

	if (msg->Buffer.data()[22]==0x00 && msg->Buffer.data()[23]==0x00 &&
	    msg->Buffer.data()[24]==0x00 && msg->Buffer.data()[25]==0x00) {
		AlarmType = Calendar_DateTime_SilentAlarm;
	}

	(*Debug)->Deb("   Alarm : %02i-%02i-%04i %02i:%02i:%02i\n",
			DT.Day,DT.Month,DT.Year,DT.Hour,DT.Minute,DT.Second);
	Calendar->AddDateTime(AlarmType, DT);

	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Phone_N6510::PrivGetCalendar3(GSM_CalendarEntry *Entry)
{
	unsigned char	   Buff[] = {
		NOKIA_FRAME1,0x7D,0x00,0x00,0x00,0x00,
		0x00,0x99,	      // Location
		0xff,0xff,0xff,0xff};

	Buff[8] = Entry->Location / 256;
	Buff[9] = Entry->Location % 256;

	(*Debug)->Deb("SENT: getting calendar note method 3\n");
	return Write(Buff, sizeof(Buff), 0x13, 4, ID_GetCalendar+ID,Entry);
}

GSM_Error GSM_Phone_N6510::GetNextCalendar3(GSM_CalendarEntry *Entry, BOOLEAN start, int *Current, int *Max)
{
	GSM_Error error;

	if (start == TRUE) {
		error = GetCalendarInfo3();
		if (error.Code != GSM_ERR_NONE) return error;
		CurrentCalendarNumber = 0;
	}

	Entry->ClearAll();

	if (CurrentCalendarNumber >= CalendarLocations.Length) {
		CalendarLocations.Locations.clear();
		return GSM_Return_Error(GSM_ERR_EMPTY);
	}

	Entry->Location = CalendarLocations.Locations.data()[CurrentCalendarNumber];

	CurrentCalendarNumber++;

	*Current = CurrentCalendarNumber;
	*Max = CalendarLocations.Length;

	return PrivGetCalendar3(Entry);
}

GSM_Error GSM_Phone_N6510::FindCalendarNoteType3(GSM_CalendarEntry *Entry, int *NoteID)
{
//GSM_Calendar_Type NoteType
	GSM_CalendarEntry	Calendar;
//	GSM_DateTime	    	DT;
	GSM_Error	       	error;
	GSM_Cal_Loc	     	CalendarLocations2;
	unsigned int		i,j;
	BOOLEAN		 	found;

	for (j=0;j<5;j++) {
		if (CalendarIcons[j].NoteType == Entry->Type) {
			(*NoteID) = CalendarIcons[j].NoteID;
			return GSM_Return_Error(GSM_ERR_NONE);
		}
	}

	error = GetCalendarInfo3();
	if (error.Code != GSM_ERR_NONE) return error;

	for (i=0;i<CalendarLocations.Locations.length();i++) {
		CalendarLocations2.Locations.push_back(CalendarLocations.Locations.data()[i]);
	}

	if ((*Phones)->Feature("cal62")==TRUE || (*Phones)->Feature("cal65")==TRUE ||
	    (*Phones)->Feature("cal35")==TRUE) {
		error = DCT34->AddCalendar2(Entry);
	} else {
		/* First method 1 was used for meeting only
		 * but it made problems with 6230 RH-12 4.44
		 * (probably for other Series 40 2.0 phones too)
		 * For now meeting, call and memo uses method 1
		 * Please note, that method 1 is the oldest one and in some
		 * moment Nokia can remove it from firmware
		 */
		if (Entry->Type == Calendar_Type_Meeting || 
		    Entry->Type == Calendar_Type_Call    ||
		    Entry->Type == Calendar_Type_Memo) {
			error = DCT34->AddCalendar1(Entry);
		} else {
			error = DCT34->AddCalendar2(Entry);
		}	       
	}
	if (error.Code != GSM_ERR_NONE) return error;

	error = GetCalendarInfo3();
	if (error.Code != GSM_ERR_NONE) return error;

	for(i=0;i<CalendarLocations.Locations.length();i++) {
		found = TRUE;
		for(j=0;j<CalendarLocations2.Locations.length();j++) {
			if (CalendarLocations.Locations.data()[i] == CalendarLocations2.Locations.data()[j]) {
				found = FALSE;
				break;
			}
		}
		if (found == TRUE) {
			Calendar.Location = CalendarLocations.Locations.data()[i];
			error = PrivGetCalendar3(&Calendar);
			if (error.Code != GSM_ERR_NONE) return error;

			error = DeleteCalendar(&Calendar);
			if (error.Code != GSM_ERR_NONE) return error;

			for (j=0;j<5;j++) {
				if (CalendarIcons[j].NoteType == Entry->Type) {
					(*NoteID) = CalendarIcons[j].NoteID;
					return GSM_Return_Error(GSM_ERR_NONE);
				}
			}

			return GSM_Return_Error(GSM_ERR_UNKNOWN);
		}
	}
	return GSM_Return_Error(GSM_ERR_UNKNOWN);
}

GSM_Error GSM_Phone_N6510::ReplyGetFirstCalPos3(GSM_Protocol_Message *msg, DebugInfo **Debug, unsigned char *S)
{
	int *Location = (int *)S;

	(*Debug)->Deb("RECEIVED: first free calendar location (method 3) is %i\n",msg->Buffer.data()[8]*256 + msg->Buffer.data()[9]);
	*Location = msg->Buffer.data()[8]*256 + msg->Buffer.data()[9];
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Phone_N6510::GetFirstCalendarPos3(int *Location)
{
	unsigned char Buff[] = {NOKIA_FRAME1, 0x95, 0x00};

	(*Debug)->Deb("SENT: Getting first free calendar note location (method 3)\n");
	return Write(Buff, sizeof(Buff), 0x13, 2, ID_GetFirstCalendarPos+ID, Location);
}

GSM_Error GSM_Phone_N6510::ReplyAddCalendar3(GSM_Protocol_Message *msg, DebugInfo **Debug, unsigned char *S)
{
	return GSM_Return_Error(GSM_ERR_NONE);
}

GSM_Error GSM_Phone_N6510::AddCalendar3(GSM_CalendarEntry *Entry)
{
	int		     Loc;
	GSM_DateTime	    Date,Date2;
	time_t		  timet1, timet2;
	unsignedstring	  Buffer,Buffer2;
	GSM_CalendarSubEntry    *SubEntry, *Text = NULL, *Location = NULL;
	GSM_CalendarSubEntry    *Phone = NULL, *DateStart = NULL, *Alarm = NULL, *DateEnd = NULL;
	int		     RepeatEach = -1, RepeatDOW = -1, RepeatDay = -1, RepeatMonth = -1;
	unsigned int	    diff;
	GSM_Error	       error;
	unsigned char	   req[54] = {
		NOKIA_FRAME1, 0x65,
		0x00,				   /* 0 = calendar, 1 = todo	       */
		0x00, 0x00, 0x00,
		0x00, 0x00,			     /* location			     */
		0x00, 0x00, 0x00, 0x00,
		0xFF, 0xFF, 0xFF, 0xFF,		 /* alarm				*/
		0x80, 0x00, 0x00,
		0x01,				   /* note icon			    */
		0xFF, 0xFF, 0xFF, 0xFF,		 /* alarm type			   */
		0x00,				   /* 0x02 or 0x00			 */
		0x01,				   /* note type			    */
		0x07, 0xD0, 0x01, 0x12, 0x0C, 0x00,     /* start date/time		      */
		0x07, 0xD0, 0x01, 0x12, 0x0C, 0x00,     /* end date/time			*/
		0x00, 0x00,			     /* recurrance			   */
		0x00, 0x00,			     /* birth year			   */
		0x20,				   /* ToDo priority			*/
		0x00,				   /* ToDo completed ?		     */
		0x00, 0x00,			     /* How many times repeat cal note or 0  */
		0x00,
		0x00,				   /* note text length		     */
		0x00,				   /* phone length/meeting place	   */
		0x00, 0x00, 0x00};
	int NoteID;

	SubEntry = NULL;
	while (Entry->GetNext(&SubEntry) == TRUE) {
		switch (SubEntry->GetType()) {
		case Calendar_Text_Text:
			Text = SubEntry;
			break;
		case Calendar_Text_Phone:
			Phone = SubEntry;
			break;
		case Calendar_Text_Location:
			Location = SubEntry;
			break;
		case Calendar_DateTime_Start:
			DateStart = SubEntry;
			break;
		case Calendar_DateTime_End:
			DateEnd = SubEntry;
			break;
		case Calendar_DateTime_ToneAlarm:
		case Calendar_DateTime_SilentAlarm:
			Alarm = SubEntry;
			break;
		case Calendar_Int_Repeat_Frequency:
			RepeatEach = SubEntry->GetInt();
			break;
		case Calendar_Int_Repeat_DayOfWeek:
			RepeatDOW = SubEntry->GetInt();
			break;		  
		case Calendar_Int_Repeat_Day:
			RepeatDay = SubEntry->GetInt();
			break;
		case Calendar_Int_Repeat_Month:
			RepeatMonth = SubEntry->GetInt();
			break;
		}
	}

	if (DateStart == NULL) return GSM_Return_Error(GSM_ERR_UNKNOWN);

	error = FindCalendarNoteType3(Entry, &NoteID);
	if (error.Code!=GSM_ERR_NONE) return error;
	req[21] = NoteID;

	error=GetFirstCalendarPos3(&Loc);
	if (error.Code!=GSM_ERR_NONE) return error;
	req[8] = Loc/256;
	req[9] = Loc%256;

	switch(Entry->Type) {
		case Calendar_Type_Reminder : req[27]=0x00; req[26]=0x02; break;
		case Calendar_Type_Meeting  : req[27]=0x01; break;
		case Calendar_Type_Call     : req[27]=0x02; break;
		case Calendar_Type_Birthday : req[27]=0x04; break;
		case Calendar_Type_Memo     : req[27]=0x08; break;
		default		     : return GSM_Return_Error(GSM_ERR_NOT_SUPPORTED);
	}

	memcpy(&Date,DateStart->GetDateTime(),sizeof(GSM_DateTime));
	req[28] = Date.Year / 256;
	req[29] = Date.Year % 256;
	req[30] = Date.Month;
	req[31] = Date.Day;
	req[32] = Date.Hour;
	req[33] = Date.Minute;

	if (Entry->Type == Calendar_Type_Birthday) {
		error = GetDateTime(&Date2);
		switch (error.Code) {
			case GSM_ERR_EMPTY:
			case GSM_ERR_NOT_SUPPORTED:
//			      GSM_GetCurrentDateTime(&Date2);
				return error;
				break;
			case GSM_ERR_NONE:
				break;
			default:
				return error;
		}
		req[28] = Date2.Year / 256;
		req[29] = Date2.Year % 256;
		if (Date.Year == 0) {
			req[42] = 0xff;
			req[43] = 0xff;
		} else {
			req[42] = Date.Year / 256;
			req[43] = Date.Year % 256;
		}
	}

	if (DateEnd != NULL) memcpy(&Date,DateEnd->GetDateTime(),sizeof(GSM_DateTime));

	req[34] = Date.Year / 256;
	req[35] = Date.Year % 256;
	req[36] = Date.Month;
	req[37] = Date.Day;
	req[38] = Date.Hour;
	req[39] = Date.Minute;
	if (Entry->Type == Calendar_Type_Birthday) {
		req[34] = Date2.Year / 256;
		req[35] = Date2.Year % 256;
	}

	//recurrance
	NokiaSetCalendarRecurranceRepeat(*Debug, Entry, req+40, req+52);

	if (Alarm != NULL) {
		memcpy(&Date,DateStart->GetDateTime(),sizeof(GSM_DateTime));

		if (Alarm->GetType() == Calendar_DateTime_SilentAlarm) {
			req[22] = 0x00; req[23] = 0x00; req[24] = 0x00; req[25] = 0x00;
		}
		if (Entry->Type == Calendar_Type_Birthday) Date.Year = Date2.Year;
		timet2   = GSMDateTime2TimeT(&Date);
		timet1   = GSMDateTime2TimeT(Alarm->GetDateTime()); 
		diff     = (timet2-timet1)/60;

		req[14] = diff / (256*256*256);
		req[15] = diff / (256*256);
		req[16] = diff / 256;
		req[17] = diff % 256;
	}

	if (Text != NULL) {
		req[49] = UnicodeLength(Text->GetText());
		NokiaSetUnicodeString(Text->GetText(),&Buffer);
	}

	if (Phone != NULL && Entry->Type == Calendar_Type_Call) {
		req[50] = UnicodeLength(Phone->GetText());
		NokiaSetUnicodeString(Phone->GetText(),&Buffer);
	}

	if (Location != NULL && Entry->Type == Calendar_Type_Meeting) {
		if ((*Phones)->Feature("cal62")==TRUE || (*Phones)->Feature("cal65")==TRUE ||
		    (*Phones)->Feature("cal35")==TRUE) {
		} else {
			req[50] = UnicodeLength(Location->GetText());
			NokiaSetUnicodeString(Location->GetText(),&Buffer);
		}
	}

	Buffer.push_back(0x00);

	Buffer2.append((const unsigned char*)req,sizeof(req));
	Buffer2.append((const unsigned char*)Buffer.data(),Buffer.size());

	(*Debug)->Deb("SENT: Adding calendar note (method 3)\n");
	return Write((unsigned char *)Buffer2.data(), Buffer2.size(), 0x13, 4, ID_AddCalendar+ID,Entry);
}

GSM_Error GSM_Phone_N6510::AddCalendar(GSM_CalendarEntry *Entry)
{
	if ((*Phones)->Feature("cal62")==TRUE) {
		return DCT34->AddCalendar1(Entry);
	} else {
		return AddCalendar3(Entry);
	}

//      return AddCalendar3(Entry);
//      return DCT34->AddCalendar2(Entry);
//      return DCT34->AddCalendar1(Entry);
}

GSM_Error GSM_Phone_N6510::DeleteCalendar(GSM_CalendarEntry *Entry)
{
	return DCT34->DeleteCalendar(Entry);
}
